
/**
 *-----------------------------------------------------------------------------------
 *    Filename: ADFAlgnZonesBAZOutlines.c
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Copyright 2004-2007 Mitsubishi Electric Research Laboratories (MERL)
 *    An implementation for BAZ grid fitting of outline-based glyphs
 *    Eric Chan, Ron Perry, and Ariel Shamir
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    The documentation in this file uses the term "real-valued coordinates" to refer
 *    to real numbers such as 0, 1.7, 3/11, and the FS_CONSTant PI. In the floating point
 *    implementation, real-valued coordinates are represented using the ADF_F32
 *    floating point data type. In the fixed point implementation, real-valued
 *    coordinates are represented using the ADF_I1616 fixed point data type.
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    Algorithm Overview
 *-----------------------------------------------------------------------------------
 *-----------------------------------------------------------------------------------
 *    The input to the BAZ alignment zone detection and grid fitting algorithm is a
 *    sequence of pen commands representing an outline-based glyph and ppem, the number
 *    of pixels per em used to render the glyph (i.e., the scale of the glyph). The
 *    coordinates of the pen commands must be specified in non-negative real-valued
 *    image coordinates.
 *
 *    The following definitions and terms are organized into several categories to
 *    facilitate comprehension.
 *
 *    Point terminology:
 *
 *      - A "point" is a two-dimensional set of coordinates (x,y) and is represented by
 *        the MAZOutlinePoint data structure. An "on-curve point" is a point that lies
 *        on the outline of an outline-based glyph. An "off-curve point" is a point
 *        that may or may not lie on the outline of an outline-based glyph, such as the
 *        control point for a quadratic Bezier curve segment.
 *
 *      - Let P be a point. The "original x coordinate" and the "original y coordinate"
 *        of P are the x and y coordinates of P in real-valued image coordinates before
 *        grid fitting is applied, respectively. The "hinted x coordinate" and the
 *        "hinted y coordinate" of P are the x and y coordinates of P in real-valued
 *        image coordinates after grid fitting is applied, respectively.
 *
 *      - The "absolute difference" between two numerical values a and b is |a - b|
 *        (i.e., the absolute value of the difference (a - b)).
 *
 *      - The "original x-distance" between two points P1 and P2 is the absolute
 *        difference between the original x coordinate of P1 and the original x
 *        coordinate of P2. The "original y-distance" between two points P1 and P2 is
 *        the absolute difference between the original y coordinate of P1 and the
 *        original y coordinate of P2.
 *
 *    Contour and radical terminology:
 *
 *      - A "contour" C represents an ordered sequence of points P1, P2, ... P{n}
 *        (where n >= 2) that define a closed path in an outline-based glyph. Contours
 *        are represented by the MAZOutlineContour data structure. Point P{i}
 *        "precedes" point P{j} if i < j. Similarly, P{i} "follows" P{j} if i > j. Two
 *        points P{i} and P{i+1} are "consecutive". P{i} "immediately precedes" P{i+1}
 *        and P{i+1} "immediately follows" P{i}.
 *
 *      - An "outer contour" is a contour that defines the outer boundary of a filled
 *        region of an outline-based glyph (e.g., the outer contour in the letter 'O').
 *        An "inner contour" is a contour that defines the inner boundary of a filled
 *        region of an outline-based glyph (e.g., the inner contour in the letter 'O').
 *
 *      - The "bounding box" of a non-degenerate contour C is the two-dimensional
 *        rectangle with the smallest area (computed using original coordinates, not
 *        hinted coordinates) that contains all of the points comprising C. The
 *        "bounding box" of a degenerate contour C (i.e., a contour whose points are
 *        collinear) is the shortest line segment that contains all of the points
 *        comprising C.
 *
 *      - A contour C is "oriented clockwise" if the points that define C's closed path
 *        follow a clockwise direction when traversing the closed path in sequential
 *        order. Similarly, C is "oriented counterclockwise" if the points that define
 *        C's closed path follow a counterclockwise direction when traversing the
 *        closed path in sequential order.
 *
 *      - Let P and C be contours. P is the "parent" of C if and only if P is the
 *        contour with the smallest bounding box (measured by area) that contains C's
 *        bounding box. If P exists, then C is a "child" of P. If no such P exists,
 *        then C has no parent contour. Contours that have no parent contours are
 *        called "root contours". All other contours (i.e., contours that do have
 *        parent contours) are called "internal contours". A contour may have at most
 *        one parent contour.
 *
 *      - Let A and D be contours. D is a "descendant" of A if and only if (1) D is a
 *        child of A or (2) there exists some ordered sequence of contours C1, C2, ...
 *        C{n} (where n >= 1) such that D is a child of C1, C{i} is a child of C{i+1}
 *        (where i is an integer that lies in the range [1,n-1]), and C{n} is a child
 *        of A. If D is a descendant of A, then A is an "ancestor" of D.
 *
 *      - A "radical" is a set of contours R, C1, C2, ... C{n} (where n >= 0) defined
 *        by a single root contour R and all of R's descendants (i.e., each C{i} is a
 *        descendant of R). Note that the term "radical" should not be confused with
 *        the concept of radicals in the Chinese language; the term has a completely
 *        different meaning as used in this file and in ADFAlgnZonesBAZ.h.
 *
 *    Simple segment terminology:
 *
 *      - A "horizontal simple segment" H consists of two consecutive points P1 and P2
 *        in a contour such that P1's original y coordinate is approximately equal to
 *        P2's original y coordinate. The original y coordinates of P1 and P2 are
 *        considered approximately equal if their absolute difference is less than
 *        BAZ_OUTLINE_EQUAL_EPS units in real-valued image coordinates;
 *        BAZ_OUTLINE_EQUAL_EPS is a macro defined below. P1 is called the "first
 *        point" in H and P2 is called the "second point" in H.
 *
 *      - A "vertical simple segment" V consists of two consecutive points P1 and P2 in
 *        a contour such that P1's original x coordinate is approximately equal to P2's
 *        original x coordinate. The original x coordinates of P1 and P2 are considered
 *        approximately equal if their absolute difference is less than
 *        BAZ_OUTLINE_EQUAL_EPS units in real-valued image coordinates;
 *        BAZ_OUTLINE_EQUAL_EPS is a macro defined below. P1 is called the "first
 *        point" in V and P2 is called the "second point" in V.
 *
 *      - The general term "simple segment" refers to either a horizontal simple
 *        segment or a vertical simple segment. Simple segments are represented by the
 *        MAZOutlineSegment data structure.
 *
 *      - A horizontal simple segment H is oriented from "left to right" if the
 *        original x coordinate of H's first point is less than the original x
 *        coordinate of H's second point. H is oriented from "right to left" if the
 *        original x coordinate of H's first point is greater than or equal to the
 *        original x coordinate of H's second point. H is "increasing" if H is oriented
 *        from left to right and "decreasing" if H is oriented from right to left.
 *
 *      - A vertical simple segment V is oriented from "bottom to top" if the original
 *        y coordinate of V's first point is less than the original y coordinate of V's
 *        second point. V is oriented from "top to bottom" if the original y coordinate
 *        of V's first point is greater than or equal to the original y coordinate of
 *        V's second point. V is "increasing" if V is oriented from bottom to top and
 *        "decreasing" if V is oriented from top to bottom.
 *
 *      - Two horizontal simple segments H1 and H2 have "opposite orientations" if
 *        either (1) H1 is oriented from left to right and H2 is oriented from right to
 *        left or (2) H1 is oriented from right to left and H2 is oriented from left to
 *        right. Two vertical simple segments V1 and V2 have "opposite orientations" if
 *        either (1) V1 is oriented from bottom to top and V2 is oriented from top to
 *        bottom or (2) V1 is oriented from top to bottom and V2 is oriented from
 *        bottom to top.
 *
 *      - The "length" of a horizontal simple segment H is the absolute difference
 *        between the original x coordinate of H's first point and the original x
 *        coordinate of H's second point. The "length" of a vertical simple segment V
 *        is the absolute difference between the original y coordinate of V's first
 *        point and the original y coordinate of V's second point.
 *
 *      - The "original slope" of a simple segment S is the ratio dy/dx, where dy = y2
 *        - y1, dx = x2 - x1, y1 is the original y coordinate of S's first point, y2 is
 *        the original y coordinate of S's second point, x1 is the original x
 *        coordinate of S's first point, and x2 is the original x coordinate of S's
 *        second point. Note that a vertical simple segment whose first point and
 *        second point have identical original x coordinates has an undefined original
 *        slope (because the denominator dx is zero).
 *
 *    Merged segment terminology:
 *
 *      - A "horizontal merged segment" consists of one or more distinct horizontal
 *        simple segments H1, H2, ... H{n} (where n >= 1) that satisfy all three of the
 *        following conditions: (1) H1 ... H{n} all have the same orientation, (2) H1
 *        ... H{n} are all contained in the same radical, and (3) the original y
 *        coordinates of the first points in H{i} and H{j} (where i and j are any
 *        integers that lie in the range [1,n]) differ by less than (2 *
 *        BAZ_OUTLINE_EQUAL_EPS) units in real-valued image coordinates. Since the
 *        original y coordinates of the first and second points in a horizontal simple
 *        segment differ by less than BAZ_OUTLINE_EQUAL_EPS units in real-valued image
 *        coordinates, condition (3) implies that the original y coordinates of the
 *        second points in H{i} and H{j} (where i and j are any integers that lie in
 *        the range [1,n]) differ by less than (3 * BAZ_OUTLINE_EQUAL_EPS) units in
 *        real-valued image coordinates.
 *
 *      - A "vertical merged segment" consists of one or more distinct vertical simple
 *        segments V1, V2, ... V{n} (where n >= 1) that satisfy all three of the
 *        following conditions: (1) V1 ... V{n} all have the same orientation, (2) V1
 *        ... V{n} are all contained in the same radical, and (3) the original x
 *        coordinates of the first points in V{i} and V{j} (where i and j are any
 *        integers that lie in the range [1,n]) differ by less than (2 *
 *        BAZ_OUTLINE_EQUAL_EPS) units in real-valued image coordinates. Since the
 *        original x coordinates of the first and second points in a vertical simple
 *        segment differ by less than BAZ_OUTLINE_EQUAL_EPS units in real-valued image
 *        coordinates, condition (3) implies that the original x coordinates of the
 *        second points in V{i} and V{j} (where i and j are any integers that lie in
 *        the range [1,n]) differ by less than (3 * BAZ_OUTLINE_EQUAL_EPS) units in
 *        real-valued image coordinates.
 *
 *      - The general term "merged segment" refers to either a horizontal merged
 *        segment or a vertical merged segment. Merged segments are represented
 *        implicitly by the MAZOutlineSegment data structure.
 *
 *      - A point P is contained in a merged segment M if (1) P is contained in a
 *        simple segment S and (2) S is contained in M.
 *
 *      - Let H be a horizontal merged segment whose minimum original x coordinate over
 *        all points in H is xMin and whose maximum original x coordinate over all
 *        points in H is xMax. The "length" of H is the difference between xMax and
 *        xMin (i.e., xMax - xMin). Let V be a vertical merged segment whose minimum
 *        original y coordinate over all points in V is yMin and whose maximum original
 *        y coordinate over all points in V is yMax. The "length" of V is the
 *        difference between yMax and yMin (i.e., yMax - yMin).
 *
 *      - The "original slope" of a merged segment M is the set of original slopes s1,
 *        s2, ... s{n} where n >= 1 and s{i} is the original slope of the ith simple
 *        segment in M.
 *
 *      - Let H1 and H2 be horizontal merged segments with lengths L1 and L2,
 *        respectively. Let L be the maximum of L1 and L2. Informally, the "overlap
 *        ratio" between H1 and H2 is the amount of horizontal overlap between H1 and
 *        H2 divided by the length of the longer segment. More formally, the "overlap
 *        ratio" between H1 and H2 is defined as (MIN(xMax1, xMax2) - MAX(xMin1,
 *        xMin2)) / L, where xMin1 is the minimum original x coordinate over all points
 *        in H1, xMin2 is the minimum original x coordinate over all points in H2,
 *        xMax1 is the maximum original x coordinate over all points in H1, and xMax2
 *        is the maximum original x coordinate over all points in H2.
 *
 *      - Let V1 and V2 be vertical merged segments with lengths L1 and L2,
 *        respectively. Let L be the maximum of L1 and L2. Informally, the "overlap
 *        ratio" between V1 and V2 is the amount of vertical overlap between V1 and V2
 *        divided by the length of the longer segment. More formally, the "overlap
 *        ratio" between V1 and V2 is defined as (MIN(yMax1, yMax2) - MAX(yMin1,
 *        yMin2)) / L, where yMin1 is the minimum original y coordinate over all points
 *        in V1, yMin2 is the minimum original y coordinate over all points in V2,
 *        yMax1 is the maximum original y coordinate over all points in V1, and yMax2
 *        is the maximum original y coordinate over all points in V2.
 *
 *      - The "first point" in a merged segment M is the first point in the first
 *        simple segment in M. In this implementation, merged segments are represented
 *        as linked lists of simple segments; the first simple segment in M is the
 *        first element of M's linked list of simple segments.
 *
 *    Segment pair terminology:
 *
 *      - A "horizontal segment pair" is a pair of two distinct horizontal merged
 *        segments H1 and H2 (written as {H1,H2}) such that (1) H1 and H2 are contained
 *        in the same radical, (2) H1 and H2 have opposite orientations, and (3) the
 *        original y coordinate of H1's first point is less than the original y
 *        coordinate of H2's first point. H1 is called the "minimum paired segment" of
 *        the segment pair {H1,H2} and H2 is called the "maximum paired segment" of the
 *        segment pair {H1,H2}. H1 and H2 are "paired" with each other.
 *
 *      - A "vertical segment pair" is a pair of two distinct vertical merged segments
 *        V1 and V2 (written as {V1,V2}) such that (1) V1 and V2 are contained in the
 *        same radical, (2) V1 and V2 have opposite orientations, and (3) the original
 *        x coordinate of V1's first point is less than the original x coordinate of
 *        V2's first point. V1 is called the "minimum paired segment" of the segment
 *        pair {V1,V2} and V2 is called the "maximum paired segment" of the segment
 *        pair {V1,V2}. V1 and V2 are "paired" with each other.
 *
 *      - The general term "segment pair" refers to either a horizontal segment pair or
 *        a vertical segment pair. Segment pairs are represented implicitly by the
 *        MAZOutlineSegment data structure.
 *
 *      - The term "paired segment" refers to either one of the two merged segments in
 *        a segment pair.
 *
 *      - An "unpaired segment" is either (1) a horizontal merged segment that is not
 *        paired with any other horizontal merged segment or (2) a vertical merged
 *        segment that is not paired with any other vertical merged segment.
 *
 *      - Let H be a horizontal merged segment whose minimum original y coordinate over
 *        all points in H is yMin and whose maximum original y coordinate over all
 *        points in H is yMax. The "original middle y coordinate" of H is the average
 *        of yMin and yMax (i.e., 0.5 * (yMin + yMax)).
 *
 *      - Let V be a vertical merged segment whose minimum original x coordinate over
 *        all points in V is xMin and whose maximum original x coordinate over all
 *        points in V is xMax. The "original middle x coordinate" of V is the average
 *        of xMin and xMax (i.e., 0.5 * (xMin + xMax)).
 *
 *      - The "first pair width" of a horizontal segment pair {H1,H2} is the difference
 *        between the original y coordinate of H2's first point and the original y
 *        coordinate of H1's first point. The "first pair width" of a vertical segment
 *        pair {V1,V2} is the difference between the original x coordinate of V2's
 *        first point and the original x coordinate of V1's first point.
 *
 *      - The "original pair width" of a horizontal segment pair {H1,H2} is the
 *        difference between H2's original middle y coordinate and H1's original middle
 *        y coordinate. The "original pair width" of a vertical segment pair {V1,V2} is
 *        the difference between V2's original middle x coordinate and V1's original
 *        middle x coordinate.
 *
 *      - A segment pair P1 is "thicker" than a segment pair P2 if P1's first pair
 *        width is greater than P2's first pair width. P1 is "thinner" than P2 if P1's
 *        first pair width is less than P2's first pair width.
 *
 *    Grid fitting terminology:
 *
 *      - A "half-integer" is an element n of the set of real numbers such that n = k +
 *        0.5 for some integer k (e.g., -2.5, -1.5, -0.5, 0.5, 1.5, 2.5, and 3.5 are
 *        all half-integers).
 *
 *      - The "integer pixel grid" is the set of ordered pairs (x,y) of real numbers x
 *        and y such that at least one element of (x,y) is an integer. For example,
 *        (0,0), (1.5,0), and (1.97,3) lie on the integer pixel grid, but (1.7,11.2)
 *        does not. Geometrically, the integer pixel grid is a grid comprising the set
 *        of integer horizontal lines (e.g., y=-2, y=-1, y=0, y=1, and y=2) and the set
 *        of integer vertical lines (e.g., x=-2, x=-1, x=0, x=1, and x=2).
 *
 *      - The "half-integer pixel grid" is the set of ordered pairs (x,y) of real
 *        numbers x and y such that at least one element of (x,y) is a half-integer.
 *        For example, (0,0.5), (1.5,0), and (1.97,2.5) lie on the half-integer pixel
 *        grid, but (1.7,11.2) does not. Geometrically, the half-integer pixel grid is
 *        a grid comprising the set of half-integer horizontal lines (e.g., y=-2.5,
 *        y=-1.5, y=-0.5, y=0.5, y=1.5, and y=2.5) and the set of half-integer vertical
 *        lines (e.g., x=-2.5, x=-1.5, x=-0.5, x=0.5, x=1.5, and x=2.5).
 *
 *    Collision terminology:
 *
 *      - Two horizontal segment pairs H1 = {A1,B1} and H2 = {A2,B2} "overlap" if there
 *        exists a vertical line V such that V intersects either A1 or B1 and V
 *        intersects either A2 or B2. Similarly, two vertical segment pairs V1 =
 *        {A1,B1} and V2 = {A2,B2} "overlap" if there exists a horizontal line H such
 *        that H intersects either A1 or B1 and H intersects either A2 or B2.
 *
 *      - The "original coordinate of a horizontal segment pair" H = {H1,H2} is the
 *        original middle y coordinate of H1. Similarly, the "original coordinate of a
 *        vertical segment pair" V = {V1,V2} is the original middle x coordinate of V1.
 *
 *      - The "hinted coordinate of a horizontal segment pair" H = {H1,H2} is the y
 *        coordinate of H1 in real-valued image coordinates after grid fitting has been
 *        applied. The "hinted coordinate of a vertical segment pair" V = {V1,V2} is
 *        the x coordinate of V1 in real-valued image coordinates after grid fitting
 *        has been applied.
 *
 *      - Let P1 and P2 be segment pairs with original coordinates c1 and c2,
 *        respectively. P1 and P2 are in "ascending order" if c1 <= c2 and in
 *        "descending order" if c1 >= c2. The terms "ascending order" and "descending
 *        order" apply generally to any number of segment pairs (e.g., 4 segment
 *        pairs).
 *
 *      - Let P1, P2, and P3 be segment pairs in ascending order. P2 lies "between" P1
 *        and P3 if and only if (1) P1 overlaps P2 and (2) P2 overlaps P3.
 *
 *      - A segment pair C is a "child" or "child segment pair" of segment pair P if
 *        and only if all three of the following conditions are satisfied:
 *
 *           a. C's original coordinate <= P's original coordinate,
 *
 *           b. C overlaps P, and
 *
 *           c. no segment pair lies between C and P.
 *
 *      - Segment pair P is the "parent" or "parent segment pair" of segment pair C if
 *        and only if C is the child of P. If P has multiple child segment pairs, the
 *        child segment pairs are referred to as the "children" of P.
 *
 *      - A segment pair D is a "descendant" of segment pair P if and only if (1) D is
 *        a child of P or (2) there exists some ordered sequence of segment pairs P1,
 *        P2, ... Pn (where n >= 1) such that D is a child of P1, P{i} is a child of
 *        P{i+1} (where 1 <= i < n), and Pn is a child of P. If D is a descendant of P,
 *        then P is an "ancestor" of D.
 *
 *      - A "segment pair tree" is the set of segment pairs defined by a segment pair P
 *        and all descendants of P. The segment pair P is the "root segment pair" of
 *        the segment pair tree.
 *
 *      - A segment pair P is "rounded up" if P's hinted coordinate h is equal to the
 *        minimum integer greater than P's original coordinate o minus 0.5 (i.e.,
 *        segment pair P is rounded up if h == floor(o - 0.5) + 1). Similarly, P is
 *        "rounded down" if P's hinted coordinate h is equal to the maximum integer not
 *        greater than P's original coordinate o minus 0.5 (i.e., segment pair P is
 *        rounded down if h == floor(o - 0.5)). Note that the Saffron library performs
 *        rasterization with samples located at pixel centers. Therefore, the algorithm
 *        described below performs grid fitting by aligning coordinates to
 *        half-integers (e.g., 0.5, 1.5, and 2.5). The unorthodox definitions of
 *        "rounded up" and "rounded down" are due to this algorithm's use of integer
 *        coordinates (instead of half-integer coordinates) to optimize performance and
 *        to maximize code re-use between floating point and fixed point
 *        implementations (see Steps 8 through 11 below).
 *
 *      - A segment pair tree is rounded up if every segment pair in the segment pair
 *        tree is rounded up. Similarly, a segment pair tree is rounded down if every
 *        segment pair in the segment pair tree is rounded down.
 *
 *      - A segment pair P1 is "aligned" to another segment pair P2 if P1 has the same
 *        hinted coordinate as P2. 
 *
 *      - Two segment pairs P1 and P2 "collide" if (1) P1 and P2 overlap and (2) their
 *        hinted coordinates are exactly 1 unit apart in real-valued image coordinates.
 *      
 *    Anchor point terminology:
 *
 *      - If P is a point that is contained in a horizontal paired segment H, then P is
 *        a "y-aligned" point. Otherwise, P is a "y-free" point.
 *
 *      - If P is a point that is contained in a vertical paired segment V, then P is
 *        an "x-aligned" point. Otherwise, P is an "x-free" point.
 *
 *      - Let C be a contour consisting of a sequence of points P1, P2, ... P{n} where
 *        n >= 2. Let P{i} be an x-free point:
 *
 *          - The "previous x anchor point" of P{i} is the first x-aligned point found
 *            during a backwards traversal through the sequence of points starting from
 *            P{i} and wrapping around from P1 to P{n} (i.e., P{i}, P{i-1}, ..., P2,
 *            P1, P{n}, P{n-1}, ... P{i+1}). Note that if C does not contain any
 *            x-aligned points, then the previous x anchor point for P{i} does not
 *            exist.
 *
 *          - The "next x anchor point" of P{i} is the first x-aligned point found
 *            during a forward traversal through the sequence of points starting from
 *            P{i} and wrapping around from P{n} to P1 (i.e., P{i}, P{i+1}, ...,
 *            P{n-1}, P{n}, P1, P2, ... P{i-1}). Note that if C does not contain any
 *            x-aligned points, then the next x anchor point for P{i} does not exist.
 *
 *      - Let C be a contour consisting of a sequence of points P1, P2, ... P{n} where
 *        n >= 2. Let P{i} be a y-free point:
 *
 *          - The "previous y anchor point" of P{i} is the first y-aligned point found
 *            during a backwards traversal through the sequence of points starting from
 *            P{i} and wrapping around from P1 to P{n} (i.e., P{i}, P{i-1}, ..., P2,
 *            P1, P{n}, P{n-1}, ... P{i+1}). Note that if C does not contain any
 *            y-aligned points, then the previous y anchor point for P{i} does not
 *            exist.
 *
 *          - The "next y anchor point" of P{i} is the first y-aligned point found
 *            during a forward traversal through the sequence of points starting from
 *            P{i} and wrapping around from P{n} to P1 (i.e., P{i}, P{i+1}, ...,
 *            P{n-1}, P{n}, P1, P2, ... P{i-1}). Note that if C does not contain any
 *            y-aligned points, then the next y anchor point for P{i} does not exist.
 *
 *      - The "original x-ratio" between an x-free point P, its previous x anchor point
 *        A1, and its next x anchor point A2 is the ratio (px - x1) / (x2 - x1), where
 *        px is P's original x coordinate, x1 is A1's original x coordinate, and x2 is
 *        A2's original x coordinate. Note that the original x-ratio is undefined if
 *        either of P's anchor points does not exist or if (x2 - x1) is zero.
 *
 *      - The "original y-ratio" between a y-free point P, its previous y anchor point
 *        A1, and its next y anchor point A2 is the ratio (py - y1) / (y2 - y1), where
 *        py is P's original y coordinate, y1 is A1's original y coordinate, and y2 is
 *        A2's original y coordinate. Note that the original y-ratio is undefined if
 *        either of P's anchor points does not exist or if (y2 - y1) is zero.
 *
 *    Linked listed terminology:
 *
 *      - Two types of linked lists are used in this implementation: linear linked
 *        lists and circular linked lists. Unless otherwise stated, the term "linked
 *        list" refers to a linear linked list.
 *
 *      - The "head" and "tail" of a linear linked list L are the first and last
 *        elements of L, respectively.
 *
 *      - Let L be a linear linked list. An element E that is not currently in L can be
 *        "prepended" to L by making E the new head of L. Similarly, E can be
 *        "appended" to L by making E the new tail of L.
 *      
 *    The following general notes apply to the algorithm described below:
 *
 *      - The algorithm is invoked dynamically during rendering. Therefore, the
 *        algorithm is designed to achieve a favorable balance between high quality,
 *        small memory consumption, and computational efficiency.
 *
 *      - The algorithm can be invoked on any outline-based glyph, but it is designed
 *        and optimized for CJK glyphs.
 *
 *      - CJK outline-based glyphs are largely comprised of horizontal and vertical
 *        features. Therefore, the algorithm detects horizontal and vertical features
 *        of the glyph and aligns them to the sampling grid. These features are called
 *        "segment pairs" in this documentation (see the Segment Pair terminology
 *        section above for a precise definition). The algorithm does not make any
 *        effort to align other features (e.g., curves or diagonal lines) to the
 *        sampling grid. Coordinates that are not aligned to the sampling grid are
 *        instead hinted using interpolation (see below for details). This overall
 *        strategy balances the competing goals of high quality and high runtime
 *        performance.
 *
 *      - Performing automatic grid fitting for CJK outline-based glyphs is challenging
 *        because of two competing goals: contrast and consistency. High contrast can
 *        be achieved by aligning both segments of a segment pair to the sampling grid,
 *        thereby producing a horizontal or vertical feature with two sharp edges. High
 *        consistency can be achieved by preserving the original pair widths of segment
 *        pairs during grid fitting so that horizontal and vertical features appear to
 *        have consistent stroke weights when rendered into the final image. These two
 *        goals compete because aligning both segments of a segment pair to the
 *        sampling grid generally requires modifying the original pair width;
 *        similarly, preserving the original pair width generally prevents both
 *        segments of a segment pair from being aligned to the sampling grid
 *        simultaneously.
 *
 *        Experiments show that consistency is more important to the overall visual
 *        quality of a rendered glyph than contrast, especially when multiple glyphs
 *        are viewed together on a single page of text. In particular, it is often
 *        preferable to accept a slight reduction in contrast (e.g., by aligning only
 *        one of the two segments in a segment pair to the sampling grid, instead of
 *        aligning both segments to the sampling grid) in exchange for a significant
 *        increase in consistency (e.g., by preserving the original pair widths of the
 *        segment pairs).
 *
 *        For this reason, the algorithm generally preserves the original pair widths
 *        of segment pairs during grid fitting, with the following exception. Segment
 *        pairs that are relatively thin (less than 1 pixel) are difficult to see when
 *        rendered. Therefore, the algorithm increases the original pair widths of thin
 *        segment pairs by a ppem-dependent amount, thereby increasing the visibility
 *        and contrast of thin features without a discernible compromise in
 *        consistency.
 *
 *        Note that aligning both segments of a segment pair to the sampling grid
 *        generally leads to disastrous results (obvious inconsistencies) at small to
 *        moderate ppems (30 ppems and below), but can lead to slightly better results
 *        at large ppems (80 ppems and above). However, even at large ppems, there are
 *        cases where consistency issues may arise in individual glyphs. For this
 *        reason, the algorithm never explicitly aligns both segments of a segment
 *        pair to the sampling grid.
 *
 *      - The algorithm attempts to determine the orientation of the outer contours in
 *        an outline-based glyph (see Step 2 below). This information enables the
 *        algorithm to detect segment pairs accurately in subsequent steps. The
 *        algorithm assumes that all outer contours in the outline-based glyph have the
 *        same orientation. If the outline-based glyph does not have this property, the
 *        algorithm may not detect segment pairs accurately.
 *
 *      - The algorithm organizes contours into groups called "radicals". Note that the
 *        term "radical" should not be confused with the concept of radicals in the
 *        Chinese language; the term as used here and in ADFAlgnZonesBAZ.h has a
 *        completely different meaning. Intuitively, a radical is a group of
 *        spatially-related components of an outline-based glyph (see the Contour and
 *        Radical terminology section above for a formal definition). The algorithm
 *        performs feature detection separately and independently for each radical;
 *        this approach improves the accuracy of feature detection by preventing
 *        segment pairs from forming between spatially-distant regions of the glyph.
 *        Note that there are no dependencies between radicals during feature
 *        detection; consequently, hardware and multi-core implementations can perform
 *        feature detection within radicals completely in parallel.
 *
 *      - The algorithm detects and resolves "collisions" between segment pairs,
 *        thereby preventing important features from becoming visually
 *        indistinguishable (see below for details). Collision resolution plays a
 *        significant role in the quality of the results. The algorithm uses a greedy
 *        method to resolve collisions. This method is computationally efficient but
 *        may not always find the optimal grid fitting configuration. In practice,
 *        however, this method produces high-quality results across a wide range of
 *        glyphs.
 *
 *      - Collision resolution is designed and optimized for small to moderate ppems
 *        (30 ppems and below). For most outline-based glyphs, segments pairs are
 *        approximately 1 pixel wide (or less) at small to moderate ppems.
 *        Consequently, collisions are defined above assuming pair widths of 1 pixel
 *        (i.e., two overlapping segment pairs "collide" if their hinted coordinates
 *        are exactly 1 pixel apart). Collision resolution is rarely needed at larger
 *        ppems (above 30 ppems) because more pixels are available to represent each
 *        feature of the glyph clearly.
 *
 *      - During collision resolution, the algorithm does not attempt to preserve the
 *        original spacing or the original spacing proportions between segments pairs.
 *
 *      - Ultimately, grid fitting requires modifying the original coordinates of the
 *        pen commands. To minimize the distortion of the glyph's shape, the algorithm
 *        never moves a coordinate more than 1.5 pixels away from its original value.
 *
 *      - The algorithm grid fits horizontal segment pairs and vertical segment pairs
 *        separately and independently. Since the algorithm contains no dependencies
 *        between horizontal segment pairs and vertical segment pairs, hardware and
 *        multi-core implementations can perform grid fitting on horizontal segment
 *        pairs and grid fitting on vertical segment pairs completely in parallel.
 *
 *      - The Saffron library follows the TrueType convention for rendering glyphs,
 *        where rasterization is performed using samples located at pixel centers.
 *        Therefore, this algorithm performs grid fitting by aligning coordinates to
 *        half-integers (e.g., 0.5, 1.5, and 2.5). To optimize performance and to
 *        maximize code re-use between floating point and fixed point implementations,
 *        Steps 8 through 11 of the algorithm (see below) perform grid fitting using
 *        the integer pixel grid. Adjustments to the integer hinted coordinates to
 *        accommodate the half-integer sampling grid are delayed until Step 12 of the
 *        algorithm (see below).
 *
 *    Algorithm description:
 *
 *      1. Determine points and contours from the input pen commands. Points and
 *         contours are represented explicitly by the MAZOutlinePoint and
 *         MAZOutlineContour data structures, respectively.
 *
 *      2. Determine whether outer contours in the glyph are oriented clockwise or
 *         counterclockwise. This determination is made by the following steps:
 *
 *           a. Determine pMinY, the point in the glyph with the minimum original y
 *              coordinate. Note that pMinY must be contained in an outer contour C of
 *              the glyph.
 *
 *           b. Compute the signed area A of the triangle defined by the three points
 *              (pMinY, pMinYN, and pMinYP), where pMinYN is the point immediately
 *              following pMinY and pMinYP is the point immediately preceding pMinY.
 *
 *           c. If A is positive, C is oriented clockwise; jump to Step (h).
 *
 *           d. If A is negative, C is oriented counterclockwise; jump to Step (h).
 *
 *           e. A is zero or approximately zero and therefore the three points (pMinY,
 *              pMinYN, and pMinYP) are collinear or approximately collinear. If pMinYN
 *              lies to the left of pMinYP, C is oriented clockwise; jump to Step (h).
 *
 *           f. If pMinYN lies to the right of pMinYP, C is oriented counterclockwise;
 *              jump to Step (h).
 *
 *           g. pMinYN and pMinYP have identical original x coordinates, so the glyph
 *              is degenerate around pMinY. It is impossible to determine C's
 *              orientation by examining the three points (pMinY, pMinYN, and pMinYP).
 *              Simply assume that C's orientation is clockwise.
 *
 *           h. Done. The computed orientation of C is assumed to be the orientation of
 *              all outer contours in the glyph.
 *
 *      3. Determine radicals (which are represented implicitly by the
 *         MAZOutlineContour data structure). A radical is a set of contours containing
 *         a single root contour and all of its descendents. Radicals are determined
 *         using the following two-step algorithm:
 *
 *           a. Determine the parent of each contour. This step effectively creates a
 *              directed acyclic graph (DAG) of contours in which each contour is
 *              linked to its parent. Note that root contours do not have parents
 *              (i.e., they are the "topmost" elements in the DAG).
 *
 *           b. Determine the radical that contains each contour. Each contour I has a
 *              unique ancestor R that is a root contour. Determine R by repeatedly
 *              following I's parent link (i.e., I->parent, I->parent->parent, etc.)
 *              until a contour is found that has no parent; this contour is R. This
 *              step effectively "flattens" the DAG hierarchy into two layers with root
 *              contours in the "top" layer and internal contours in the "bottom"
 *              layer.
 *
 *      4. Determine horizontal simple segments and vertical simple segments by
 *         traversing the points determined above in Step 1 and by applying the
 *         definitions of a horizontal simple segment and a vertical simple segment
 *         described above.
 *
 *      5. For each radical R, merge the horizontal simple segments in R into
 *         horizontal merged segments by applying the definition of a horizontal merged
 *         segment described above. Similarly, for each radical R, merge the vertical
 *         simple segments in R into vertical merged segments by applying the
 *         definition of a vertical merged segment described above. Merged segments are
 *         represented implicitly by the MAZOutlineSegment data structure using linked
 *         lists of simple segments. There is no significance to the ordering of simple
 *         segments within these linked lists.
 *
 *      6. For each radical R, determine horizontal segment pairs from R's horizontal
 *         merged segments and determine vertical segment pairs from R's vertical
 *         merged segments.
 *
 *         Horizontal segment pairs are determined using a brute-force algorithm that
 *         considers all possible horizontal segment pairs {H1,H2}, where H1 and H2 are
 *         horizontal merged segments in R. {H1,H2} is a valid segment pair if and only
 *         if all of the following conditions are satisfied:
 *
 *           - {H1,H2} must span a filled region of the glyph. This occurs if either
 *             (1) H1 is oriented from left to right and the outer contours of the
 *             glyph are oriented counterclockwise or (2) H1 is oriented from right to
 *             left and the outer contours of the glyph are oriented clockwise. The
 *             orientation of the outer contours of the glyph is determined above in
 *             Step 2.
 *
 *           - The first pair width of {H1,H2} must be at most
 *             (BAZ_OUTLINE_MAX_WIDTH_FRAC * ppem) units in real-valued image
 *             coordinates, where ppem is the user-specified number of pixels per em
 *             used to render the glyph. BAZ_OUTLINE_MAX_WIDTH_FRAC is a macro defined
 *             below.
 *
 *           - The overlap ratio between H1 and H2 must be at least
 *             BAZ_OUTLINE_MIN_OVERLAP_RATIO, where BAZ_OUTLINE_MIN_OVERLAP_RATIO is a
 *             macro defined below.
 *
 *           - Let w be the first pair width of {H1,H2}. There cannot exist a
 *             horizontal merged segment H3 in R such that the first pair width of
 *             {H3,H2} is less than w. Similarly, there cannot exist a horizontal
 *             merged segment H3 in R such that the first pair width of {H1,H3} is less
 *             than w. Intuitively, {H1,H2} must be the thinnest horizontal segment
 *             pair over all possible horizontal segment pairs containing either H1 or
 *             H2.
 *
 *         Vertical segment pairs are determined in the same manner as for horizontal
 *         segment pairs, substituting x coordinates for y coordinates.
 *
 *      7. For each radical R, append each horizontal segment pair in R to an
 *         uninitialized horizontal segment pair array of MAZOutlineSegPair data
 *         structures. Similarly, for each radical R, append each vertical segment pair
 *         in R to an uninitialized vertical segment pair array of MAZOutlineSegPair
 *         data structures.
 *
 *      8. Determine the hinted coordinate of each horizontal segment pair in the
 *         horizontal segment pair array by computing the floor of its original
 *         coordinate. Similarly, determine the hinted coordinate of each vertical
 *         segment pair in the vertical segment pair array by computing the floor of
 *         its original coordinate. Note that the hinted coordinates of the segment
 *         pairs determined in this step may be modified below to resolve collisions
 *         (see Step 11).
 *
 *      9. Sort the horizontal segment pairs in the horizontal segment pair array into
 *         ascending order (i.e., sort by non-decreasing original coordinates).
 *         Similarly, sort the vertical segment pairs in the vertical segment pair
 *         array into ascending order (i.e., sort by non-decreasing original
 *         coordinates).
 *
 *     10. Identify the child segment pairs of each horizontal segment pair in the
 *         horizontal segment pair array and build a directed acyclic graph (DAG).
 *         Similarly, identify the child segment pairs of each vertical segment pair in
 *         the vertical segment pair array and build a DAG. The DAGs are represented
 *         implicitly by the MAZOutlineSegPair data structure by linking each segment
 *         pair to its child segment pairs.
 *
 *         The purpose of building these DAGs is to detect and resolve collisions
 *         between segment pairs in the next step. A collision between a segment pair P
 *         and a child segment pair C cannot occur during collision resolution if the
 *         current hinted coordinates of P and C differ by more than 3 pixels.
 *         Therefore, to improve runtime efficiency, a link in the DAG is added between
 *         P and C if and only if their hinted coordinates differ by 3 pixels or less.
 *
 *     11. Detect colliding segment pairs and resolve collisions that occur in Step 8
 *         using a greedy bottom-up algorithm. Two overlapping segment pairs collide if
 *         their hinted coordinates lie exactly 1 pixel apart. Visually, colliding
 *         segment pairs appear to be a single thick segment pair (instead of two
 *         separate segment pairs). Collisions can be resolved in some cases by adding
 *         +1 or -1 to the hinted coordinate of one of the segment pairs so that the
 *         two segment pairs become at least 2 pixels apart (see details below).
 *
 *         Collisions are resolved using a greedy algorithm in which each DAG of
 *         segment pairs determined in the previous step is traversed in a bottom-up
 *         order (i.e., from the bottommost descendants to the topmost ancestors).
 *         Since the segment pairs are sorted in ascending order (see Step 9 above),
 *         this bottom-up traversal is accomplished by visiting each segment pair in
 *         the segment pair array, beginning with the first segment pair in the segment
 *         pair array and ending with the last segment pair in the segment pair array.
 *         For each segment pair P visited during this bottom-up traversal, apply the
 *         following steps for each child C of P that collides with P:
 *
 *           a. If C is rounded up and can be rounded down without colliding with any
 *              of its children, then round down C and jump to Step (e). (Example: if
 *              C's original coordinate is 3.4, then C's hinted coordinate is normally
 *              "rounded up" to 3.0. "Rounding down" instead produces the integer 2.0.
 *              Refer to the definitions of "rounded up" and "rounded down" in the
 *              Collision terminology section above.) This step attempts to eliminate a
 *              collision and always avoids introducing any new collisions.
 *
 *           b. If C is rounded up and the segment pair tree with root C can be rounded
 *              down without creating any collisions whatsoever in the segment pair
 *              tree, then round down the segment pair tree and jump to Step (e). This
 *              step attempts to eliminate a collision and always avoids introducing
 *              any new collisions.
 *
 *           c. If P is rounded down, then round up P and jump to Step (e). (Example:
 *              if P's original coordinate is 5.8, then P's hinted coordinate is
 *              normally rounded down to 5.0 (i.e., floor(5.8 - 0.5) = floor(5.3) =
 *              5.0). Rounding up instead produces the integer 6.0.) This step attempts
 *              to eliminate a collision but may introduce a collision between P and a
 *              parent of P. This potential collision will be treated when processing
 *              the parent of P in a subsequent iteration of the bottom-up traversal of
 *              the DAG.
 *
 *           d. Align P to C. (Example: if C's hinted coordinate is 4.0 and P's hinted
 *              coordinate is 5.0, then align P to 4.0.)
 *
 *           e. Done.
 *
 *         Steps (a) through (c) attempt to round segment pairs C and P so that they
 *         are at least 2 pixels apart. However, there are some cases in which
 *         collisions cannot be resolved by this method. Since C and P cannot be
 *         visually distinguished in these cases, Step (d) aligns P to C, thereby
 *         simplifying the appearance of the glyph by aligning two segment pairs to the
 *         same coordinate.
 *
 *     12. Grid fit each horizontal segment pair {S1,S2} in the horizontal segment pair
 *         array:
 *
 *           a. Determine the original pair width w of {S1,S2} in real-valued image
 *              coordinates.
 *
 *           b. Determine the adjusted pair width w' of {S1,S2} as follows:
 *
 *                - Initialize w' to w.
 *
 *                - If w' is less than 0.5, set w' to 0.5.
 *
 *                - If w' is less than 1.0, set w' = w' + f * (1 - w'), where f is the
 *                  pair width adjustment factor. f is computed as (ppem * 0.0625 -
 *                  0.875) and clamped to the range [0,1]. ppem is the user-specified
 *                  number of pixels per em. This step increases the original pair
 *                  width by a ppem-dependent amount to increase the visibility and
 *                  contrast of thin segment pairs.
 *
 *              The pair width adjustment factor (i.e., f) lies in the range [0,1] and
 *              behaves as a linear ramp from 0 (when ppem = 14) to 1 (when ppem = 30).
 *              Therefore, the pair width adjustment factor is minimized at 14 ppems
 *              and maximized at 30 ppems. These values were determined empirically to
 *              achieve a good balance between contrast and consistency.
 *
 *           c. Grid fit the points of S1 and S2 as follows:
 *
 *                - Let omy1 and omy2 be the original middle y coordinates of S1 and
 *                  S2, respectively. The hinted y coordinate hy1 of S1 was determined
 *                  earlier by Step 8 and Step 11 (see above).
 *
 *                - Determine the hinted y coordinate hy2 of S2 by adding the adjusted
 *                  pair width (i.e., w') to the hinted y coordinate of S1 (i.e., hy1).
 *
 *                - For each point P contained in S1, determine P's hinted y coordinate
 *                  by adding (hy1 - omy1 + 0.5) to P's original y coordinate. This
 *                  step effectively preserves the original slope of S1. The additional
 *                  delta of 0.5 causes P to be aligned to the half-integer pixel grid.
 *
 *                - For each point P contained in S2, determine P's hinted y coordinate
 *                  by adding (hy2 - omy2 + 0.5) to P's original y coordinate. This
 *                  step effectively preserves the original slope of S2. The additional
 *                  delta of 0.5 causes P to be aligned to the half-integer pixel grid.
 *
 *         Grid fit each vertical segment pair in the vertical segment pair array using
 *         the same method described above for horizontal segment pairs, substituting x
 *         coordinates for y coordinates.
 *         
 *     13. For each radical R, perform grid fitting on the x-free points in R:
 *
 *           a. Let P be an x-free point in R. Determine P's previous x anchor point A1
 *              and P's next x anchor point A2. Note that P may not have any anchor
 *              points (i.e., neither A1 nor A2 exist).
 *
 *           b. If neither of P's anchor points exist, then jump to Step (g). Note that
 *              P will either have exactly zero anchor points or exactly two anchor
 *              points.
 *
 *           c. Determine the original x-ratio xr between P, A1, and A2. If xr is
 *              undefined (i.e., A1 and A2 have identical original x coordinates and
 *              therefore the denominator of xr is zero), then determine P's hinted x
 *              coordinate by preserving the original x-distance between P and A1 and
 *              jump to Step (g).
 *
 *           d. If xr is less than zero, then (1) P's original x coordinate is closer
 *              to A1's original x coordinate than to A2's original x coordinate and
 *              (2) P's original x coordinate does not lie between the original x
 *              coordinates of the anchor points. In this case, determine P's hinted x
 *              coordinate by preserving the original x-distance between P and A1 and
 *              jump to Step (g).
 *
 *           e. If xr is greater than 1, then (1) P's original x coordinate is closer
 *              to A2's original x coordinate than to A1's original x coordinate and
 *              (2) P's original x coordinate does not lie between the original x
 *              coordinates of the anchor points. In this case, determine P's hinted x
 *              coordinate by preserving the original x-distance between p and A2 and
 *              jump to Step (g).
 *
 *           f. xr lies in the range [0,1] and therefore P's original x coordinate (1)
 *              is equal to A1's original x coordinate, (2) is equal to A2's original x
 *              coordinate, or (3) lies between the original x coordinates of the
 *              anchor points. Determine P's hinted x coordinate by linearly
 *              interpolating between the hinted x coordinates of the anchor points.
 *
 *           g. Done.
 *
 *         For each radical R, perform grid fitting on the y-free points in R using the
 *         same algorithm as described above for x-free points, substituting y
 *         coordinates for x coordinates.
 *
 *     14. Copy the hinted x and y coordinates from points to the input pen commands,
 *         thereby overwriting the original x and y coordinates of the input pen
 *         commands.
 *-----------------------------------------------------------------------------------
 */


/**
 *-----------------------------------------------------------------------------------
 *    Required include files for this implementation
 *-----------------------------------------------------------------------------------
 */
#include <stdlib.h>
#include <string.h>

#include "fs_object.h"
#include "fs_function.h"

/**
 *-----------------------------------------------------------------------------------
 *    START: iType ADF Rendering
 *-----------------------------------------------------------------------------------
 */
#ifdef FS_EDGE_HINTS

/*#include "adfinittermsystem.h" */ /* not needed */
#include "adfalgnzonesmaz.h"


/**
 *-----------------------------------------------------------------------------------
 *    START: IMPLICIT ADFS ONLY
 *-----------------------------------------------------------------------------------
 */
#if (ADF_USE_IMPLICIT_ADFS)


/**
 *-----------------------------------------------------------------------------------
 *    START: BAZ ALGN ZONE DETECTION AND GRID FITTING FOR OUTLINE-BASED GLYPHS ONLY
 *-----------------------------------------------------------------------------------
 */
#if (1)


/**
 *-----------------------------------------------------------------------------------
 *    START: FIXED POINT MATH ONLY
 *-----------------------------------------------------------------------------------
 */

/**
 *-----------------------------------------------------------------------------------
 *    FS_CONSTants for the ADF_I1616 fixed point values of 0, 1, 0.5, -1, and 0.2
 *-----------------------------------------------------------------------------------
 */
#define BAZ_OUTLINE_ZERO               (0)
#define BAZ_OUTLINE_ONE                (I1616_CONST_1)
#define BAZ_OUTLINE_TWO                (0x20000)
#define BAZ_OUTLINE_ONE_HALF           (I1616_CONST_ONE_HALF)
#define BAZ_OUTLINE_NEG_ONE            (-I1616_CONST_1)


/**
 *-----------------------------------------------------------------------------------
 *    A small 'epsilon' ADF_I1616 fixed point value (determined empirically) for
 *    avoiding numerical errors in the function BAZDetermineContourOrientation() (see
 *    below). BAZ_OUTLINE_EPS has the mathematical value 64.0 / 65536.0 = 0.0009765625
 *    and can be represented exactly as a floating point value and as an ADF_I1616
 *    fixed point value.
 *-----------------------------------------------------------------------------------
 */
#define BAZ_OUTLINE_EPS                   (0x40)


/**
 *-----------------------------------------------------------------------------------
 *    Two consecutive points A and B whose original x coordinates (in real-valued image
 *    coordinates) differ by at most BAZ_OUTLINE_EQUAL_EPS define a vertical simple
 *    segment. Two consecutive points A and B whose original y coordinates (in
 *    real-valued image coordinates) differ by at most BAZ_OUTLINE_EQUAL_EPS define a
 *    horizontal simple segment. BAZ_OUTLINE_EQUAL_EPS has the mathematical value
 *    8192.0 / 65536.0 = 0.125 and can be represented exactly as a floating point value
 *    and as an ADF_I1616 fixed point value.
 *
 *    The value of BAZ_OUTLINE_EQUAL_EPS was determined empirically. A non-zero value
 *    was chosen to accommodate glyphs with nearly (but not exactly) horizontal
 *    features or nearly (but not exactly) vertical features.
 *-----------------------------------------------------------------------------------
 */
#define BAZ_OUTLINE_EQUAL_EPS           (0x2000)
#define BAZ_OUTLINE_EQUAL_EPS3          (0x8000)


/**
 *-----------------------------------------------------------------------------------
 *    The minimum overlap ratio between two paired segments, expressed as an ADF_I1616
 *    fixed point value. Two paired segments in a horizontal segment pair must overlap
 *    in the x direction by at least (BAZ_OUTLINE_MIN_OVERLAP_RATIO * 100)%. Two paired
 *    segments in a vertical segment pair must overlap in the y direction by at least
 *    (BAZ_OUTLINE_MIN_OVERLAP_RATIO * 100)%. BAZ_OUTLINE_MIN_OVERLAP_RATIO has the
 *    mathematical value 2048.0 / 65536.0 = 0.03125 and can be represented exactly as a
 *    floating point value and as an ADF_I1616 fixed point value.
 *-----------------------------------------------------------------------------------
 */
#define BAZ_OUTLINE_MIN_OVERLAP_RATIO  (0x0800)


/**
 *-----------------------------------------------------------------------------------
 *    A fractional value (expressed as an ADF_I1616 fixed point value) used to restrict
 *    the first pair width of segment pairs. This restriction improves the accuracy of
 *    segment pair determination. A segment pair cannot be thicker than
 *    (BAZ_OUTLINE_MAX_WIDTH_FRAC * ppem) in real-valued image coordinates, where ppem
 *    is the pixels per em used to perform grid fitting. BAZ_OUTLINE_MAX_WIDTH_FRAC has
 *    the mathematical value 6553.0 / 65536.0 = 0.0999908447265625 and can be
 *    represented exactly as a floating point value and as an ADF_I1616 fixed point
 *    value. The approximate value of this FS_CONSTant is 0.1, whose value has been
 *    determined from experiments showing that first pair widths in most CJK typefaces
 *    are no thicker than 10% of the em box.
 *-----------------------------------------------------------------------------------
 */

#define BAZ_OUTLINE_MAX_WIDTH_FRAC       (0x2100)


/**
 *-----------------------------------------------------------------------------------
 *    Round the ADF_I1616 fixed point value v to the nearest integral ADF_I1616 fixed
 *    point value
 *-----------------------------------------------------------------------------------
 */
#define BAZ_OUTLINE_ROUND_R32(v)       (((v) + I1616_CONST_ONE_HALF) & 0xffff0000)


/**
 *-----------------------------------------------------------------------------------
 *    Convert the integer v to an ADF_I1616 fixed point value
 *-----------------------------------------------------------------------------------
 */
#define BAZ_OUTLINE_I32_TO_R32(v)       ((v) << 16)


/**
 *-----------------------------------------------------------------------------------
 *    Compute and return the product of ADF_I1616 fixed point values a and b
 *-----------------------------------------------------------------------------------
 */
#define BAZ_OUTLINE_MUL(a,b)           (I1616_MUL(a, b))


/**
 *-----------------------------------------------------------------------------------
 *    Compute and return the pair width adjustment factor (in ADF_I1616 fixed point
 *    image coordinates) given p, the number of pixels per em used to perform grid
 *    fitting
 *-----------------------------------------------------------------------------------
 */
#define BAZ_OUTLINE_COMPUTE_PWAF(p)       (I1616_MUL(p, 0x1000) - 0xe000)


/**
 *-----------------------------------------------------------------------------------
 *    Compute and return the reciprocal of the ADF_I1616 fixed point value v
 *-----------------------------------------------------------------------------------
 */
ADF_INLINE static ADF_I1616 BAZ_OUTLINE_RCP (ADF_I1616 v)
{
    ADF_I32 status;
    return(I1616_DIV(I1616_CONST_1, v, &status));
}


/**
 *-----------------------------------------------------------------------------------
 *    END: FIXED POINT MATH ONLY
 *-----------------------------------------------------------------------------------
 */


/**
 *-----------------------------------------------------------------------------------
 *    Return true if the absolute difference between the original x coordinates of
 *    points a and b is less than BAZ_OUTLINE_EQUAL_EPS in real-valued image
 *    coordinates. Return false otherwise.
 *-----------------------------------------------------------------------------------
 */
#define BAZ_OUTLINE_IS_EQ_X(a,b)  (((a)->ox > (b)->ox) ? ((a)->ox - (b)->ox < \
BAZ_OUTLINE_EQUAL_EPS) : ((b)->ox - (a)->ox < BAZ_OUTLINE_EQUAL_EPS))

/* the below  macro is not currently used
#define BAZ_OUTLINE_IS_EQ_X2(a,b)  (((a)->ox > (b)->ox) ? ((a)->ox - (b)->ox < \
BAZ_OUTLINE_EQUAL_EPS2) : ((b)->ox - (a)->ox < BAZ_OUTLINE_EQUAL_EPS2))
*/

#define BAZ_OUTLINE_IS_EQ_MERGE_X(a,b)  (((a)->ox > (b)->ox) ? ((a)->ox - (b)->ox < \
BAZ_OUTLINE_EQUAL_EPS3) : ((b)->ox - (a)->ox < BAZ_OUTLINE_EQUAL_EPS3))

/* the below  macro is not currently used
#define BAZ_OUTLINE_IS_EQ(a,b)  (((a) > (b)) ? ((a) - (b) < \
BAZ_OUTLINE_EQUAL_EPS3) : ((b) - (a) < BAZ_OUTLINE_EQUAL_EPS3))
*/

/**
 *-----------------------------------------------------------------------------------
 *    Return true if the absolute difference between the original y coordinates of
 *    points a and b is less than BAZ_OUTLINE_EQUAL_EPS in real-valued image
 *    coordinates. Return false otherwise.
 *-----------------------------------------------------------------------------------
 */

#define BAZ_OUTLINE_IS_EQ_Y(a,b)  (((a)->oy > (b)->oy) ? ((a)->oy - (b)->oy < \
BAZ_OUTLINE_EQUAL_EPS) : ((b)->oy - (a)->oy < BAZ_OUTLINE_EQUAL_EPS))

/* the below  macro is not currently used
#define BAZ_OUTLINE_IS_EQ_Y2(a,b)  (((a)->oy > (b)->oy) ? ((a)->oy - (b)->oy < \
BAZ_OUTLINE_EQUAL_EPS2) : ((b)->oy - (a)->oy < BAZ_OUTLINE_EQUAL_EPS2))
*/

#define BAZ_OUTLINE_IS_EQ_MERGE_Y(a,b)  (((a)->oy > (b)->oy) ? ((a)->oy - (b)->oy < \
BAZ_OUTLINE_EQUAL_EPS3) : ((b)->oy - (a)->oy < BAZ_OUTLINE_EQUAL_EPS3))

/* the below  macro is not currently used
#define BAZ_OUTLINE_IS_TOOBIG(a,b,max) (((a) > (b)) ? ((a) - (b) > \
(max)) : ((b) - (a) > (max)))
*/



/**
 *-----------------------------------------------------------------------------------
 *    Return true if the bounding box of the contour big contains the bounding box of
 *    the contour sm. Return false otherwise.
 *-----------------------------------------------------------------------------------
 */
#define BAZ_OUTLINE_BBOX_CONTAINS(big, sm)   (((big)->xMax >= (sm)->xMax) && \
                                             ((big)->yMax >= (sm)->yMax) && \
                                             ((big)->xMin <= (sm)->xMin) && \
                                             ((big)->yMin <= (sm)->yMin))


/**
 *-----------------------------------------------------------------------------------
 *    Compute and return the signed area (in real-valued original image coordinates)
 *    defined by the triangle with the specified point P and the edge vectors A and B,
 *    where A is the vector from P to P's previous point (i.e., P->prev) and B is the
 *    vector from P to P's next point (i.e., P->next). BAZComputeNeighborSignedArea()
 *    effectively computes and returns the cross product of vectors A and B.
 *-----------------------------------------------------------------------------------
 */
static ADF_INLINE ADF_R32 BAZComputeNeighborSignedArea (MAZOutlinePoint *p)
{
    ADF_R32 dx0 = p->prev->ox - p->ox;
    ADF_R32 dy0 = p->prev->oy - p->oy;
    ADF_R32 dx1 = p->next->ox - p->ox;
    ADF_R32 dy1 = p->next->oy - p->oy;

    return(BAZ_OUTLINE_MUL(dx0, dy1) - BAZ_OUTLINE_MUL(dx1, dy0));
}

/**
 *-----------------------------------------------------------------------------------
 *    Convert the pen commands in the specified pen command array penCmds to
 *    MAZOutlinePoints and MAZOutlineContours and store the MAZOutlinePoints and
 *    MAZOutlineContours in the specified point array points and the specified contour
 *    array contours, respectively.
 *
 *    Upon entry, penCmds is the contiguous array of numPenCmds pen commands to be
 *    converted to MAZOutlinePoints and MAZOutlineContours, points is the uninitialized
 *    contiguous array of at least (2 * numPenCmds) points in which the
 *    MAZOutlinePoints will be stored, and contours is the uninitialized contiguous
 *    array of at least numPenCmds contours in which the MAZOutlineContours will be
 *    stored.
 *
 *    Upon exit, points is a valid array of MAZOutlinePoints, contours is a valid array
 *    of MAZOutlineContours, outNumPoints contains the number of valid MAZOutlinePoint
 *    elements in the points array, and outNumContours contains the number of valid
 *    MAZOutlineContour elements in the contours array.
 *
 *    BAZDeterminePointsAndContours() returns the point in the points array with the
 *    minimum original y coordinate.
 *-----------------------------------------------------------------------------------
 */
static ADF_I32 BAZDeterminePointsAndContours (fnt_LocalGraphicStateType *gs, 
        MAZOutlinePoint *points, MAZOutlineContour *contours, ADF_I32 *outNumPoints, 
        ADF_I32 *outNumContours)
{
    /**
     *----Local variables
     */
    ADF_I32 i,j;
    ADF_I32 numPoints = 0;
    ADF_I32 numContours = gs->CE0->nc;
    MAZOutlineContour *c;
    MAZOutlinePoint *pyMin = 0;
    MAZOutlinePoint *pyContourMin = 0;
    MAZOutlineContour *lowestContour = 0;
    ADF_R32 signedArea;
    /* MAZOutlinePoint *contourFirstPoint = 0; */
    


    /**
     *----Set a pointer to the first (uninitialized) point in the points array
     */
    MAZOutlinePoint *p = points;

    


    /**
     *----Set a pointer to the first (uninitialized) contour in the contours array
     */
    

    c = contours;

   

    
    /**
     *----Process each pen command in the pen command array penCmds to determine
     *----points and contours. Note that in the penCmds array, the (x,y) coordinates
     *----of the last pen command of each closed path must equal the (x,y)
     *----coordinates of the closed path's initial moveto command (i.e., the penCmds
     *----array contains a duplicate set of (x,y) coordinates for each closed path).
     *----In contrast, when FS_CONSTructing each contour in the contours array (where
     *----each contour corresponds to a closed path in the penCmds array), the (x,y)
     *----coordinates of the last pen command of the closed path are not added to the
     *----contour.
     */
    for ( i = 0; i < gs->CE0->nc; i++) {


        /**
        *----Clear all elements of the next available uninitialized
        *----MAZOutlineContour data structure to zero. This data structure is
        *----used to represent the new contour. Initialize the bounding box of
        *----the new contour and set a pointer to the first point in the new
        *----contour.
        */
        SYS_MEMSET(c, 0, sizeof(MAZOutlineContour));
            
        c->points = p;
        



        /**
        *----Set a pointer from p to the contour
        */
        p->contour = c;

        
        for (j=gs->CE0->sp[i]; j<=gs->CE0->ep[i]; j++) {


            /**
            *----Copy the x and y coordinates of penCmd to the original x and y
            *----coordinates and to the hinted x and y coordinates of the current
            *----point (i.e., p), respectively
            */
            p->hx = p->ox = gs->CE0->x[j]<<10;
            p->hy = p->oy = gs->CE0->y[j]<<10;


                

            /**
            *----Set p's flags
            */
            p->flags = 0;
            if (!gs->CE0->onCurve[j])
               p->flags = OUTLINE_OFF_CURVE;
            



            p->contour = c;


            /**
             *----Update c's bounding box
            */
            if (j == gs->CE0->sp[i]) {
                c->xMin = p->ox;
                c->xMax = p->ox;
                c->yMin = p->oy;
                c->yMax = p->oy;

                /**
                *----Initialize pyMin, the point in the points array with the minimum
                *----original y coordinate
                */
                if (!pyMin) {
                    pyMin = p;
                    lowestContour = c;
                }
                else if (p->oy < pyMin->oy) {
                    pyMin = p;
                    lowestContour = c;
                }
                else if (p->oy == pyMin->oy && p->ox < pyMin->ox) {
                    pyMin = p;
                    lowestContour = c;
                }
                

                pyContourMin = p;
            }
            else {
                if (p->ox > c->xMax)      
                    c->xMax = p->ox;
                else if (p->ox < c->xMin) 
                    c->xMin = p->ox;
                if (p->oy > c->yMax)      
                    c->yMax = p->oy;
                else if (p->oy < c->yMin) 
                    c->yMin = p->oy;
                /**
                *----Link p to its previous point and vice versa
                */
                p->prev = p - 1;
                p[-1].next = p;
                /**
                *----Update pyMin, the point in the points array with the minimum
                *----original y coordinate
                */
                if (!pyMin || p->oy < pyMin->oy) {
                    pyMin = p;
                    lowestContour = c;
                }
                else if (p->oy == pyMin->oy && p->ox < pyMin->ox) {
                    pyMin = p;
                    lowestContour = c;
                }
                

                if (pyContourMin)
                {
                    if (p->oy < pyContourMin->oy) 
                        pyContourMin = p;
                    else if (p->oy == pyContourMin->oy && p->ox < pyContourMin->ox)
                        pyContourMin = p;
                }
                else
                    pyContourMin = p;

            }


            /**
            *----Increment the total number of points in this glyph
            */
            ++numPoints;


            /**
             *----Advance p to the next (uninitialized) point in the points array
             */
            ++p;


            
        }

        /**
        *----Link the last and first points in the previous contour to each
        *----other (i.e., complete the circular linked list of points in the
        *----previous contour)
        */
        (p-1)->next = c->points;
        c->points->prev = p-1;

        /**
        *----Determine the area of the contour's bounding box
        */
        c->bboxArea = BAZ_OUTLINE_MUL(c->xMax - c->xMin, c->yMax - c->yMin);


        /**
        *---- Determine the directionn of each contour: the
        *----area of the triangle defined by the three points (pyMinContour, pMinContourY's next
        *----point, and pyContourMin's previous point). If the signed area is positive, the
        *----outer contour is oriented clockwise; if the signed area is negative, the
        *----outer contour is oriented counterclockwise. Assume black-on-right and use lowest contour
        *----to correct at the end.
        */
        if (pyContourMin)    /* should always be true */
        {
            signedArea = BAZComputeNeighborSignedArea(pyContourMin);
            if (signedArea > BAZ_OUTLINE_EPS) 
                c->isOuter = 1;
            else if (signedArea < -BAZ_OUTLINE_EPS)
                c->isOuter = 0;


            /**
            *----The signed area is zero or approximately zero and therefore the three
            *----points (pMinY, pMinY's next point, and pMinY's previous point) are
            *----collinear or nearly collinear. If pMinY's next point lies to the left of
            *----pMinY's previous point, the outer contour is oriented clockwise. If pMinY's
            *----next point lies to the right of pMinY's previous point, the outer contour
            *----is oriented counterclockwise.
            */
            else if (pyContourMin->next->ox < pyContourMin->prev->ox)
                c->isOuter = 1;
            else if (pyContourMin->next->ox > pyContourMin->prev->ox)
                c->isOuter = 0;


            /**
            *----pMinY's next point has the same original x coordinate as pMinY's previous
            *----point, which implies that the glyph is degenerate around pMinY. It is
            *----impossible to determine the orientation of the outer contour by examining
            *----the three points (pMinY, pMinY's next point, and pMinY's previous point).
            *----Assume that outer contours are oriented clockwise.
            */
            else
                c->isOuter = 1;
        }
        else
            c->isOuter = 1;
        

        c++;


        
    }

    if (!lowestContour || !lowestContour->isOuter)
    {
        /* Contours are black-on-left - reverse values of isOuter */
        c = contours;
        for ( i = 0; i < numContours; i++) 
        {
            c->isOuter = 1-c->isOuter;
            c++;
        }
    }


    /**
     *----Store the total number of points in the glyph and the total number of
     *----contours in the glyph
     */
    *outNumPoints = numPoints;
    *outNumContours = numContours;


    /**
     *----Return the point in the points array with the minimum original y coordinate
     */
    if (lowestContour)
        return(lowestContour->isOuter );
    else
        return 1;
}


/**
 *-----------------------------------------------------------------------------------
 *    Determine radicals (which are represented implicitly by the MAZOutlineContour
 *    data structure) from the contours in the specified contour array contours. Upon
 *    entry, contours is the contiguous array of numContours contours from which the
 *    radicals will be determined and contourList is an uninitialized working array of
 *    at least numContours pointers to contours. BAZDetermineRadicals() uses the
 *    contourList array internally to determine radicals. Upon exit, the determined
 *    radicals are represented implicitly in the specified contour array contours.
 *-----------------------------------------------------------------------------------
 */
static void BAZDetermineRadicals (MAZOutlineContour *contours, ADF_I32 numContours,
MAZOutlineContour **contourList)
{
    /**
     *----Uninitialized local variables
     */
    ADF_I32 i;
    MAZOutlineContour *c;
    MAZOutlineContour **pContour;

    
    /**
     *----For each contour C in the contours array, store a pointer to C in the
     *----contourList array
     */
    for (c = contours, i = 0; i < numContours; ++i, ++c) contourList[i] = c;


    /**
     *----Sort the contours in the contourList array into ascending order using the
     *----selection sort algorithm. A contour C1 is "less than" a contour C2 if the
     *----bounding box area of C1 is less than the bounding box area of C2 (i.e., the
     *----bounding box area of a contour is used as the sorting key). This
     *----implementation sorts pointers to contours instead of directly sorting
     *----contours themselves, thereby requiring only pointer swaps. Note that the
     *----number of contours is typically small (e.g., fewer than 10) and therefore
     *----an asymptotically faster sorting algorithm (e.g., quicksort) is not
     *----required.
     */
    for (pContour = contourList, i = 0; i < numContours; ++i, ++pContour) {


        /**
         *----An uninitialized local variable
         */
        ADF_I32 j;


        /**
         *----An integer array index into contourList in the range [i,numContours-1]
         *----corresponding to the contour with the minimum bounding box area in the
         *----current iteration of the selection sort algorithm
         */
        ADF_I32 minIndex = i;


        /**
         *----Initialize minArea to the bounding box area (in real-valued image
         *----coordinates) corresponding to the contour with integer array index
         *----minIndex
         */
        ADF_R32 minArea = (*pContour)->bboxArea;


        /**
         *----Set a pointer to the next contour in the contourList array
         */
        MAZOutlineContour **pContourN = pContour + 1;


        /**
         *----Process each remaining contour in the contourList array to determine
         *----the contour in the range [i,numContours-1] with the minimum bounding
         *----box area
         */
        MAZOutlineContour *tmp;
        for (j = i + 1; j < numContours; ++j, ++pContourN) {


            /**
             *----If the bounding box area of pContourN is less than minArea, update
             *----the integer array index of the contour with the smallest bounding
             *----box (i.e., minIndex) and update the minimum bounding box area
             *----(i.e., minArea)
             */
            if ((*pContourN)->bboxArea < minArea) {
                minIndex = j;
                minArea = (*pContourN)->bboxArea;
            }
        }


        /**
         *----Swap the pContour pointer with the contour pointer corresponding to the
         *----contour with the minimum bounding box area
         */
        tmp = *pContour;
        *pContour = contourList[minIndex];
        contourList[minIndex] = tmp;
    }


    /**
     *----Determine radicals from contours in a two-step algorithm. Step 1: For each
     *----contour C in the sorted contourList array, determine the parent contour of
     *----C (i.e., the contour P with the smallest bounding box that contains C's
     *----bounding box). If P exists, then temporarily set C->radical = P; otherwise,
     *----set C->radical = C. Note that the following implementation requires that
     *----the contours in the contourList array are sorted in ascending order by
     *----bounding box area as performed above.
     */
    for (i = 0, pContour = contourList; i < numContours; ++i, ++pContour) {
        ADF_I32 j;
        MAZOutlineContour **largerContour = pContour + 1;

        /*  All outer contours are their own radicals. 
         *  The smallest outer contour that contains and inner contour is the inner contour's radical
         */

        if ((*pContour)->isOuter) {
            (*pContour)->radical = *pContour;
            continue;
        }


        /**
         *----Process each contour in the contourList array whose integer array index
         *----lies in the range [i+1,numContours-1]
         */

        for (j = i + 1; j < numContours; ++j, ++largerContour) {


            /**
             *----If the bounding box of largerContour contains the bounding box of
             *----pContour, then largerContour must be the contour with the smallest
             *----bounding box that contains the bounding box of pContour (this is
             *----true because the contours are sorted in ascending order by bounding
             *----box area) and therefore largerContour must be the parent of
             *----pContour
             */
            if ( (*largerContour)->isOuter && BAZ_OUTLINE_BBOX_CONTAINS(*largerContour, *pContour)) {
                (*pContour)->radical = *largerContour;
                break;
            }
        }



        /**
         *----If no parent of pContour is found, then pContour must be a root contour
         *--- This shouldn't happen now.
         */
        if (!((*pContour)->radical)) (*pContour)->radical = *pContour;
    }



}


/**
 *-----------------------------------------------------------------------------------
 *    Determine horizontal simple segments and vertical simple segments from the points
 *    in the specified point array points. Upon entry, points is the contiguous array
 *    of numPoints points from which horizontal simple segments and vertical simple
 *    segments will be determined, hSegPtr is a pointer to an uninitialized array of at
 *    least numPoints horizontal simple segments to which the determined horizontal
 *    simple segments will be written, and vSegPtr is a pointer to an uninitialized
 *    array of at least numPoints vertical simple segments to which the determined
 *    vertical simple segments will be written.
 *-----------------------------------------------------------------------------------
 */
static void BAZDetermineSimpleSegments (MAZOutlinePoint *points, ADF_I32 numPoints,
MAZOutlineSegment *hSegPtr, MAZOutlineSegment *vSegPtr)
{
    /**
     *----Set a pointer to the first (uninitialized) horizontal simple segment of the
     *----hSegPtr array
     */
    MAZOutlineSegment *hSeg = hSegPtr;


    /**
     *----Set a pointer to the first (uninitialized) vertical simple segment of the
     *----vSegPtr array
     */
    MAZOutlineSegment *vSeg = vSegPtr;


    /**
     *----Process each point in the points array to determine horizontal simple
     *----segments and vertical simple segments
     */
    ADF_I32 j;
    MAZOutlinePoint *p = points;
    for (j = 0; j < numPoints; ++j, ++p) {

            
        /**
         *----A stroke is defined as either two on-curve points with the same coordinate
         *----or a curve tangent point, an off curve point followed by an on curve point followed by an off curve point
         */
    

                
        /**
         *----Both p and p->next are on-curve points. Determine if their original y
         *----coordinates are approximately equal.
         */
        if ( BAZ_OUTLINE_IS_EQ_Y(p, p->next) && !BAZ_OUTLINE_IS_EQ_X(p,p->next) ) 

        {

                    
            /**
             *----The original y coordinates of p and p->next are approximately
             *----equal. Therefore, a horizontal simple segment whose first point is
             *----p and whose second point is p->next has been determined. Set
             *----pointers from the horizontal simple segment (i.e., hSeg) to its
             *----first point (i.e., p) and to its second point (i.e., p->next).
             */
           hSeg->p2 = p->next;

                hSeg->p1 = p;

            

            
                


            /**
             *----Determine if hSeg is increasing (i.e., oriented from left to right)
             *----or decreasing (i.e., oriented from right to left)
             */
            hSeg->increasing = (p->next->ox > p->ox) ? 1 : 0;


            /**
             *----Initialize hSeg's isMerged Boolean to false (i.e., hSeg is
             *----currently a horizontal simple segment and has not yet been merged
             *----into a horizontal merged segment)
             */
            hSeg->isMerged = 0;



            /**
             *----Let R be the root contour corresponding to the radical that
             *----contains points p and p->next. Since R contains p and p->next, R
             *----also contains hSeg. Prepend hSeg to R's linked list of horizontal
             *----simple segments (i.e., hSeg becomes the head of R's linked list of
             *----horizontal simple segments). The order of horizontal simple
             *----segments in this linked list is not relevant to subsequent
             *----processing.
             */
            hSeg->next = p->contour->radical->hSeg;
            p->contour->radical->hSeg = hSeg;

                
            /**
             *----Initialize hSeg's remaining pointers to NULL
             */
            hSeg->mergeChain = hSeg->nextMerge = 0;
            hSeg->pairMax = hSeg->pairMin = 0;


            /**
             *----Determine the minimum and maximum original x coordinates of hSeg
             */
            if (hSeg->p1->ox < hSeg->p2->ox) {
                hSeg->minLocal = hSeg->p1->ox;
                hSeg->maxLocal = hSeg->p2->ox;
            } else {
                hSeg->minLocal = hSeg->p2->ox;
                hSeg->maxLocal = hSeg->p1->ox;
            }

                
            /**
             *----Determine the minimum and maximum original y coordinates of hSeg
             */
            if (hSeg->p1->oy < hSeg->p2->oy) {
                hSeg->minPerpLocal = hSeg->p1->oy;
                hSeg->maxPerpLocal = hSeg->p2->oy;
            } else {
                hSeg->minPerpLocal = hSeg->p2->oy;
                hSeg->maxPerpLocal = hSeg->p1->oy;
            }


            /**
             *----Advance hSeg to the next (uninitialized) horizontal simple segment
             *----in the hSegPtr array
             */
            ++hSeg;
        }

                
        /**
         *----The original y coordinates of p and p->next are not approximately equal
         *----(i.e., p and p->next do not determine a horizontal simple segment).
         *----Determine if their original x coordinates are approximately equal.
         */
        if ( BAZ_OUTLINE_IS_EQ_X(p, p->next) && !BAZ_OUTLINE_IS_EQ_Y(p,p->next))
      
        {


            /**
             *----The original x coordinates of p and p->next are approximately
             *----equal. Therefore, a vertical simple segment whose first point is p
             *----and whose second point is p->next has been determined. Set pointers
             *----from the vertical simple segment (i.e., vSeg) to its first point
             *----(i.e., p) and to its second point (i.e., p->next).
             */
           
            vSeg->p2 = p->next;

          
                vSeg->p1 = p;

                
            /**
             *----Determine if vSeg is increasing (i.e., oriented from bottom to top)
             *----or decreasing (i.e., oriented from top to bottom)
             */
            vSeg->increasing = (p->next->oy > p->oy) ? 1 : 0;


            /**
             *----Initialize vSeg's isMerged Boolean to false (i.e., vSeg is
             *----currently a vertical simple segment and has not yet been merged
             *----into a vertical merged segment)
             */
            vSeg->isMerged = 0;
            
                

            /**
             *----Let R be the root contour corresponding to the radical that
             *----contains points p and p->next. Since R contains p and p->next, R
             *----also contains vSeg. Prepend vSeg to R's linked list of vertical
             *----simple segments (i.e., vSeg becomes the head of R's linked list of
             *----vertical simple segments). The order of vertical simple segments in
             *----this linked list is not relevant to subsequent processing.
             */
            vSeg->next = p->contour->radical->vSeg;
            p->contour->radical->vSeg = vSeg;
                

            /**
             *----Initialize vSeg's remaining pointers to NULL
             */
            vSeg->mergeChain = vSeg->nextMerge = 0;
            vSeg->pairMax = vSeg->pairMin = 0;


            /**
             *----Determine the minimum and maximum original y coordinates of vSeg
             */
            if (vSeg->p1->oy < vSeg->p2->oy) {
                vSeg->minLocal = vSeg->p1->oy;
                vSeg->maxLocal = vSeg->p2->oy;
            } else {
                vSeg->minLocal = vSeg->p2->oy;
                vSeg->maxLocal = vSeg->p1->oy;
            }

                
            /**
             *----Determine the minimum and maximum original x coordinates of vSeg
             */
            if (vSeg->p1->ox < vSeg->p2->ox) {
                vSeg->minPerpLocal = vSeg->p1->ox;
                vSeg->maxPerpLocal = vSeg->p2->ox;
            } else {
                vSeg->minPerpLocal = vSeg->p2->ox;
                vSeg->maxPerpLocal = vSeg->p1->ox;
            }


            /**
             *----Advance vSeg to the next (uninitialized) vertical simple segment in
             *----the vSegPtr array
             */
            ++vSeg;
        }
    }


}




/**
 *-----------------------------------------------------------------------------------
 *    Determine horizontal merged segments and vertical merged segments from the
 *    horizontal simple segments and the vertical simple segments contained
 *    (implicitly) in the specified contour array contours, respectively. Upon entry,
 *    contours is the contiguous array of numContours contours from which the
 *    horizontal merged segments and vertical merged segments will be determined. Upon
 *    exit, the determined horizontal merged segments and vertical merged segments are
 *    represented implicitly in the specified contour array contours.
 *-----------------------------------------------------------------------------------
 */
static void BAZDetermineMergedSegments (MAZOutlineContour *contours, ADF_I32 numContours)
{
    /**
     *----Uninitialized local variables
     */
    ADF_I32 i;
    MAZOutlineSegment *tail;
    MAZOutlineSegment *hSeg;
    MAZOutlineSegment *vSeg;
    MAZOutlineSegment *hSegN;
    MAZOutlineSegment *vSegN;
    MAZOutlineSegment *hMergeSeg;
    MAZOutlineSegment *vMergeSeg;


     /**
     *----The FS_CONSTant k_maxWidth is the maximum distance (in real-valued image
     *----coordinates) allowed between the two segments in a valid segment pair
     */
    /* FS_CONST ADF_R32 k_maxWidthx = BAZ_OUTLINE_MUL(ppemx, BAZ_OUTLINE_MAX_WIDTH_FRAC) * 2; */
    /* FS_CONST ADF_R32 k_maxWidthy = BAZ_OUTLINE_MUL(ppemy, BAZ_OUTLINE_MAX_WIDTH_FRAC) * 2; */


    /**
     *----Process each root contour in the contours array (i.e., skip over internal
     *----contours). Each root contour implicitly represents a radical. Horizontal
     *----merged segments are determined separately for each radical. Similarly,
     *----vertical merged segments are determined separately for each radical.
     */
    MAZOutlineContour *c = contours;
    for (i = 0; i < numContours; ++i, ++c) 
    {


        /**
         *----Proceed to the next contour if c is an internal contour (i.e., skip
         *----internal contours)
         */
        if (c != c->radical) continue;


        /**
         *----c is a root contour and therefore (implicitly) represents a radical R.
         *----Set hSeg to the head of R's linked list of horizontal simple segments.
         */
        hSeg = c->hSeg;


        /**
         *----Set vSeg to the head of R's linked list of vertical simple segments
         */
        vSeg = c->vSeg;


        /**
         *----hMergeSeg is a pointer to the head of R's linked list of horizontal
         *----merged segments (initialized to NULL)
         */
        hMergeSeg = 0;


        /**
         *----vMergeSeg is a pointer to the head of R's linked list of vertical
         *----merged segments (initialized to NULL)
         */
        vMergeSeg = 0;

        /**
         *----Process each horizontal simple segment in R and determine horizontal
         *----merged segments
         */
        for (; hSeg; hSeg = hSeg->next) 
        {


            /**
             *----If hSeg has already been merged during an earlier iteration of this
             *----for-loop, proceed to the next horizontal simple segment
             */
            if (hSeg->isMerged) continue;



            /**
             *----Let M denote the new horizontal merged segment containing hSeg.
             *----Currently, M contains a single horizontal simple segment (i.e.,
             *----hSeg). Set a pointer to the tail of M's linked list of horizontal
             *----simple segments. hSeg is the only horizontal simple segment in M
             *----and is therefore the tail of M.
             */
            tail = hSeg;


            /**
             *----Set hSeg's isMerged Boolean to true to indicate that hSeg is
             *----contained in a horizontal merged segment (i.e., M)
             */
            hSeg->isMerged = 1;


            /**
             *----Initialize the minimum and maximum original x coordinates of M
             */
            hSeg->minMerged = hSeg->minLocal;
            hSeg->maxMerged = hSeg->maxLocal;


            /**
             *----Initialize the minimum and maximum original y coordinates of M
             */
            hSeg->minPerpMerged = hSeg->minPerpLocal;
            hSeg->maxPerpMerged = hSeg->maxPerpLocal;


            /**
             *----Set a pointer to the next horizontal simple segment in R's linked
             *----list of horizontal simple segments
             */
            hSegN = hSeg->next;




            /**
             *----Process each remaining horizontal simple segment in R to determine
             *----if there are other horizontal simple segments that can be merged
             *----into M
             */
            for (; hSegN; hSegN = hSegN->next) 
            {




                /**
                 *----Skip hSegN if it has a different orientation than hSeg (i.e.,
                 *----hSeg and hSegN can be merged if and only if they have the same
                 *----orientation)
                 */
                if (hSeg->increasing != hSegN->increasing) continue;


                /**
                 *----Skip hSegN if the original y coordinate of hSegN's first point
                 *----is not sufficiently close to the original y coordinate of
                 *----hSeg's first point (i.e., hSeg and hSegN can be merged if and
                 *----only if the original y coordinates of their first points are
                 *----approximately equal)
                 */
                if (!BAZ_OUTLINE_IS_EQ_MERGE_Y(hSeg->p1, hSegN->p1)) continue;



                /**
                 *----hSegN can be merged into M. Perform the merge by appending
                 *----hSegN to M's linked list. After this step, hSegN becomes the
                 *----new tail of M.
                 */
                tail->mergeChain = hSegN;
                tail = hSegN;


                /**
                 *----Set hSegN's isMerged Boolean to true to indicate that hSegN is
                 *----now contained in a horizontal merged segment (i.e., M)
                 */
                hSegN->isMerged = 1;



                /**
                 *----Update the minimum and maximum original x coordinates of M
                 */
                if (hSegN->minLocal < hSeg->minMerged)
                    hSeg->minMerged = hSegN->minLocal;
                if (hSegN->maxLocal > hSeg->maxMerged)
                    hSeg->maxMerged = hSegN->maxLocal;


                /**
                 *----Update the minimum and maximum original y coordinates of M
                 */
                if (hSegN->minPerpLocal < hSeg->minPerpMerged)
                    hSeg->minPerpMerged = hSegN->minPerpLocal;
                if (hSegN->maxPerpLocal > hSeg->maxPerpMerged)
                    hSeg->maxPerpMerged = hSegN->maxPerpLocal;
            }


            /**
             *----Prepend M to R's linked list of horizontal merged segments
             */
            hSeg->nextMerge = hMergeSeg;
            hMergeSeg = hSeg;
        }


        /**
         *----Set a pointer to the head of R's linked list of horizontal merged
         *----segments
         */
        c->hMergeSeg = hMergeSeg;


        /**
         *----Process each vertical simple segment in R and determine vertical merged
         *----segments
         */
        for (; vSeg; vSeg = vSeg->next) 
        {


            /**
             *----If vSeg has already been merged during an earlier iteration of this
             *----for-loop, proceed to the next vertical simple segment
             */
            if (vSeg->isMerged) continue;



            /**
             *----Let M denote the new vertical merged segment containing vSeg.
             *----Currently, M contains a single vertical simple segment (i.e.,
             *----vSeg). Set a pointer to the tail of M's linked list of vertical
             *----simple segments. vSeg is the only vertical simple segment in M and
             *----is therefore the tail of M.
             */
            tail = vSeg;


            /**
             *----Set vSeg's isMerged Boolean to true to indicate that vSeg is
             *----contained in a vertical merged segment (i.e., M)
             */
            vSeg->isMerged = 1;


            /**
             *----Initialize the minimum and maximum original y coordinates of M
             */
            vSeg->minMerged = vSeg->minLocal;
            vSeg->maxMerged = vSeg->maxLocal;


            /**
             *----Initialize the minimum and maximum original x coordinates of M
             */
            vSeg->minPerpMerged = vSeg->minPerpLocal;
            vSeg->maxPerpMerged = vSeg->maxPerpLocal;


            /**
             *----Set a pointer to the next vertical simple segment in R's linked
             *----list of vertical simple segments
             */
            vSegN = vSeg->next;




            /**
             *----Process each remaining vertical simple segment in R to determine if
             *----there are other vertical simple segments that can be merged into M
             */
            for (; vSegN; vSegN = vSegN->next) 
            {



                /**
                 *----Skip vSegN if it has a different orientation than vSeg (i.e.,
                 *----vSeg and vSegN can be merged if and only if they have the same
                 *----orientation)
                 */
                if (vSeg->increasing != vSegN->increasing) 
                    continue;


                /**
                 *----Skip vSegN if the original x coordinate of vSegN's first point
                 *----is not sufficiently close to the original x coordinate of
                 *----vSeg's first point (i.e., vSeg and vSegN can be merged if and
                 *----only if the original x coordinates of their first points are
                 *----approximately equal)
                 */
                if (!BAZ_OUTLINE_IS_EQ_MERGE_X(vSeg->p1, vSegN->p1))
                    continue;




                /**
                 *----vSegN can be merged into M. Perform the merge by appending
                 *----vSegN to M's linked list. After this step, vSegN becomes the
                 *----new tail of M.
                 */
                tail->mergeChain = vSegN;
                tail = vSegN;


                /**
                 *----Set vSegN's isMerged Boolean to true to indicate that vSegN is
                 *----now contained in a vertical merged segment (i.e., M)
                 */
                vSegN->isMerged = 1;


                /**
                 *----Update the minimum and maximum original y coordinates of M
                 */
                if (vSegN->minLocal < vSeg->minMerged)
                    vSeg->minMerged = vSegN->minLocal;
                if (vSegN->maxLocal > vSeg->maxMerged)
                    vSeg->maxMerged = vSegN->maxLocal;


                /**
                 *----Update the minimum and maximum original x coordinates of M
                 */
                if (vSegN->minPerpLocal < vSeg->minPerpMerged)
                    vSeg->minPerpMerged = vSegN->minPerpLocal;
                if (vSegN->maxPerpLocal > vSeg->maxPerpMerged)
                    vSeg->maxPerpMerged = vSegN->maxPerpLocal;
            }


            /**
             *----Prepend M to R's linked list of vertical merged segments
             */
            vSeg->nextMerge = vMergeSeg;
            vMergeSeg = vSeg;
        }


        /**
         *----Set a pointer to the head of R's linked list of vertical merged
         *----segments
         */
        c->vMergeSeg = vMergeSeg;
    }
}

static void BAZOutlineRecomputeWidthY(MAZOutlineSegPair *pairSeg)
{
    
    /* compute the width between two straight segments if there are both 
    straight and curved segments */
    MAZOutlineSegment *s = pairSeg->seg->pairMax;
    ADF_R32 top=0;
    ADF_R32 bottom;
    ADF_I32 found = 0;
            
    while(s) 
    {
   
        if ( (s->p1->flags & OUTLINE_OFF_CURVE) == 0 &&
            (s->p2->flags & OUTLINE_OFF_CURVE) == 0) 
        {
            top = s->p1->oy;
            found++;
            break;
        }
        s = s->mergeChain;
        
    } 
        
    s = pairSeg->seg->pairMax->pairMin;
    bottom = s->p1->oy;
       
    while(s) 
    {
        
        if ( (s->p1->flags & OUTLINE_OFF_CURVE) == 0 &&
            (s->p2->flags & OUTLINE_OFF_CURVE) == 0) 
        {
            bottom = s->p1->oy;   
            found++;
            break;
        }
        s = s->mergeChain;
    } 

    if (found == 2)
        pairSeg->width = top - bottom;

         
}
static void BAZOutlineRecomputeWidthX(MAZOutlineSegPair *pairSeg)
{
    /* compute the width between two straight segments if there are both 
    straight and curved segments */
    MAZOutlineSegment *s = pairSeg->seg->pairMax;
    ADF_R32 right=0;
    ADF_R32 left=0;
    ADF_I32 found = 0;
            
    while(s) 
    {
   
        if ( (s->p1->flags & OUTLINE_OFF_CURVE) == 0 &&
            (s->p2->flags & OUTLINE_OFF_CURVE) == 0) 
        {
            right = s->p1->ox;
            found++;
            break;
        }
        s = s->mergeChain;
        
    } 
        
    s = pairSeg->seg->pairMax->pairMin;
    
       
    while(s) 
    {
        
        if ( (s->p1->flags & OUTLINE_OFF_CURVE) == 0 &&
            (s->p2->flags & OUTLINE_OFF_CURVE) == 0) 
        {
            left = s->p1->ox;   
            found++;
            break;
        }
        s = s->mergeChain;
    } 

    if (found == 2)
        pairSeg->width = right - left;

}

/**
 *-----------------------------------------------------------------------------------
 *    Determine horizontal segment pairs and vertical segment pairs from the horizontal
 *    merged segments and vertical merged segments contained (implicitly) in the
 *    specified contour array contours, respectively. Upon entry, contours is the
 *    contiguous array of numContours contours from which the horizontal segments pairs
 *    and vertical segment pairs will be determined, ppem is the number of pixels per
 *    em that will be used to perform grid fitting, and outerClockwise is a Boolean set
 *    to true if outer contours in the contours array are oriented clockwise and false
 *    otherwise. Upon exit, the determined horizontal segment pairs and vertical
 *    segment pairs are represented implicitly in the specified contour array contours,
 *    outNumHSegPairs contains the number of horizontal segment pairs determined, and
 *    outNumVSegPairs contains the number of vertical segment pairs determined.
 *-----------------------------------------------------------------------------------
 */
static void BAZDetermineSegmentPairs (MAZOutlineContour *contours, ADF_I32
numContours, ADF_R32 ppemx, ADF_R32 ppemy, ADF_I32 outerClockwise, ADF_I32 *outNumHSegPairs, ADF_I32
*outNumVSegPairs)
{
    /**
     *----Local variables
     */
    ADF_I32 i;
    ADF_I32 numHSegPairs = 0;
    ADF_I32 numVSegPairs = 0;
    MAZOutlineSegment *hSeg;
    MAZOutlineSegment *vSeg;


    /**
     *----The FS_CONSTant k_overlapRatio is the minimum overlap ratio (expressed as a
     *----real-valued number) between the two segments in a valid segment pair
     */
    FS_CONST ADF_R32 k_overlapRatio = BAZ_OUTLINE_MIN_OVERLAP_RATIO;


    /**
     *----The FS_CONSTant k_maxWidth is the maximum distance (in real-valued image
     *----coordinates) allowed between the two segments in a valid segment pair
     */
    FS_CONST ADF_R32 k_maxWidthx = BAZ_OUTLINE_MUL(ppemx, BAZ_OUTLINE_MAX_WIDTH_FRAC);
    FS_CONST ADF_R32 k_maxWidthy = BAZ_OUTLINE_MUL(ppemy, BAZ_OUTLINE_MAX_WIDTH_FRAC);

    
    /**
     *----Process each root contour in the contours array (i.e., skip over internal
     *----contours). Each root contour implicitly represents a radical. Horizontal
     *----segment pairs are determined separately for each radical. Similarly,
     *----vertical segment pairs are determined separately for each radical.
     */
    MAZOutlineContour *c = contours;
    for (i = 0; i < numContours; ++i, ++c) 
    {


        /**
         *----Proceed to the next contour if c is an internal contour (i.e., skip
         *----internal contours)
         */
        if (c != c->radical) continue;



        /**
         *----Process each horizontal merged segment in R and determine horizontal
         *----segment pairs
         */
        for (hSeg = c->hMergeSeg; hSeg; hSeg = hSeg->nextMerge) 
        {


            /**
             *----An uninitialized horizontal merged segment
             */
            MAZOutlineSegment *hSeg2;


            /**
             *----The uninitialized length of hSeg
             */
            ADF_R32 hSegLength;


            /**
             *----Initialize the optimal paired segment to NULL. This pointer will
             *----eventually be set to hSeg's optimal paired segment or will remain
             *----NULL if no optimal paired segment is found.
             */
            MAZOutlineSegment *bestPairSeg = 0;


            /**
             *----Initialize the minimum first pair width to an invalid value
             */
            ADF_R32 minPairWidth = BAZ_OUTLINE_NEG_ONE;


            /**
             *----Proceed to the next horizontal merged segment if either (1) hSeg is
             *----oriented from left to right and the outer contours of the glyph are
             *----oriented clockwise or (2) hSeg is oriented from right to left and
             *----the outer contours of the glyph are oriented counterclockwise. This
             *----step prevents horizontal segment pairs from spanning unfilled
             *----regions of the glyph.
             */
            if (hSeg->increasing == outerClockwise) continue;


            /**
             *----Compute the length of hSeg
             */
            hSegLength = hSeg->maxMerged - hSeg->minMerged;


            /**
             *----Determine hSeg's optimal paired segment (note that an optimal
             *----paired segment for hSeg may not exist)
             */
            for (hSeg2 = c->hMergeSeg; hSeg2; hSeg2 = hSeg2->nextMerge)
            {


                /**
                 *----An uninitialized first pair width
                 */
                ADF_R32 width;


                /**
                 *----Skip hSeg2 if it is the same as hSeg (i.e., hSeg cannot be
                 *----paired with itself)
                 */
                if (hSeg == hSeg2) continue;


                /**
                 *----Skip hSeg2 if its orientation matches hSeg's orientation (i.e.,
                 *----hSeg can only be paired with hSeg2 if the two segments have
                 *----opposite orientations)
                 */
                if (hSeg2->increasing == hSeg->increasing) continue;


                /**
                 *----Approximate the vertical distance V between hSeg2 and hSeg by
                 *----computing the difference in the original y coordinates of
                 *----hSeg2's first point and hSeg's first point
                 */
                width = hSeg2->p1->oy - hSeg->p1->oy;



                /**
                 *----Skip hSeg2 if the vertical distance V is negative or is greater
                 *----than the maximum allowable first pair width (i.e., hSeg2 must
                 *----be above hSeg and not too distant from hSeg)
                 */
                if ((width <= BAZ_OUTLINE_ZERO) || (width > k_maxWidthy)) continue;


                /**
                 *----Skip hSeg2 if the vertical distance V is greater than the
                 *----minimum first pair width found so far (i.e., search for the
                 *----segment pair with the minimum first pair width)
                 */
                if ((minPairWidth > BAZ_OUTLINE_ZERO) && (width > minPairWidth))
                    continue;


                /**
                 *----Skip hSeg2 if it does not sufficiently overlap hSeg
                 */
                {
                    /**
                     *----Determine the horizontal overlap between hSeg and hSeg2
                     */
                    ADF_R32 overlap = ADF_MIN(hSeg->maxMerged, hSeg2->maxMerged) -
                    ADF_MAX(hSeg->minMerged, hSeg2->minMerged);


                    /**
                     *----Determine the length of hSeg2
                     */
                    ADF_R32 hSegLength2 = hSeg2->maxMerged - hSeg2->minMerged;


                    /**
                     *----Determine the maximum of hSegLength and hSegLength2
                     */
                    ADF_R32 maxLength = (hSegLength2 > hSegLength) ? hSegLength2 :
                    hSegLength;


                    /**
                     *----Skip hSeg2 if it does not sufficiently overlap hSeg (i.e.,
                     *----the two segments in a segment pair must overlap by at least
                     *----k_overlapRatio)
                     */
                    if (overlap < BAZ_OUTLINE_MUL(k_overlapRatio, maxLength))
                        continue;
                }

                

                /**
                 *----hSeg2 satisfies all of the above criteria and is the new
                 *----optimal paired segment for hSeg. Update the minimum first pair
                 *----width and the optimal paired segment.
                 */
                minPairWidth = width;
                bestPairSeg = hSeg2;
            }


            /**
             *----All horizontal merged segments in R have been considered as
             *----potential optimal paired segments for hSeg. Determine if an optimal
             *----paired segment for hSeg exists.
             */
            if (bestPairSeg) 
            {


                /**
                 *----An optimal paired segment for hSeg (i.e., bestPairSeg) exists.
                 *----It is possible that bestPairSeg has already been paired with
                 *----another horizontal merged segment M during an earlier iteration
                 *----of this for-loop. Determine whether to pair bestPairSeg with
                 *----hSeg or M by choosing the segment pair with the minimum first
                 *----pair width.
                 */
                if ((!(bestPairSeg->pairMin)) || (hSeg->p1->oy >=
                bestPairSeg->pairMin->p1->oy)) 
                {


                    /**
                     *----Either (1) bestPairSeg is an unpaired segment or (2)
                     *----bestPairSeg is the maximum paired segment in a segment pair
                     *----P1 = {M,bestPairSeg} that is thicker than the segment pair
                     *----P2 = {hSeg,bestPairSeg}. In the former case, form the
                     *----segment pair P2 and update the number of horizontal segment
                     *----pairs. In the latter case, delete the existing segment pair
                     *----P1 by setting M's pairMax pointer to NULL and form the
                     *----segment pair P2.
                     */
                    if (bestPairSeg->pairMin) bestPairSeg->pairMin->pairMax = 0;
                    else ++numHSegPairs;
                    hSeg->pairMax = bestPairSeg;
                    bestPairSeg->pairMin = hSeg;
                    

                    
                }
            }
        }

        
        /**
         *----Process each vertical merged segment in R and determine vertical
         *----segment pairs
         */
        for (vSeg = c->vMergeSeg; vSeg; vSeg = vSeg->nextMerge) 
        {


            /**
             *----An uninitialized vertical merged segment
             */
            MAZOutlineSegment *vSeg2;


            /**
             *----The uninitialized length of vSeg
             */
            ADF_R32 vSegLength;


            /**
             *----Initialize the optimal paired segment to NULL. This pointer will
             *----eventually be set to vSeg's optimal paired segment or will remain
             *----NULL if no optimal paired segment is found.
             */
            MAZOutlineSegment *bestPairSeg = 0;


            /**
             *----Initialize the minimum first pair width to an invalid value
             */
            ADF_R32 minPairWidth = BAZ_OUTLINE_NEG_ONE;


            /**
             *----Proceed to the next vertical merged segment if either (1) vSeg is
             *----oriented from bottom to top and the outer contours of the glyph are
             *----oriented counterclockwise or (2) vSeg is oriented from top to
             *----bottom and the outer contours of the glyph are oriented clockwise.
             *----This step prevents vertical segment pairs from spanning unfilled
             *----regions of the glyph.
             */
            if (vSeg->increasing != outerClockwise) continue;

            
            /**
             *----Compute the length of vSeg
             */
            vSegLength = vSeg->maxMerged - vSeg->minMerged;


            /**
             *----Determine vSeg's optimal paired segment (note that an optimal
             *----paired segment for vSeg may not exist)
             */
            for (vSeg2 = c->vMergeSeg; vSeg2; vSeg2 = vSeg2->nextMerge) 
            {


                /**
                 *----An uninitialized first pair width
                 */
                ADF_R32 width;


                /**
                 *----Skip vSeg2 if it is the same as vSeg (i.e., vSeg cannot be
                 *----paired with itself)
                 */
                if (vSeg == vSeg2) continue;

                
                /**
                 *----Skip vSeg2 if its orientation matches vSeg's orientation (i.e.,
                 *----vSeg can only be paired with vSeg2 if the two segments have
                 *----opposite orientations)
                 */
                if (vSeg2->increasing == vSeg->increasing) continue;

                
                /**
                 *----Approximate the horizontal distance H between vSeg2 and vSeg by
                 *----computing the difference in the original x coordinates of
                 *----vSeg2's first point and vSeg's first point
                 */
                width = vSeg2->p1->ox - vSeg->p1->ox;



                /**
                 *----Skip vSeg2 if the horizontal distance H is negative or is
                 *----greater than the maximum allowable first pair width (i.e.,
                 *----vSeg2 must be to the right of vSeg and not too distant from
                 *----vSeg)
                 */
                if ((width <= BAZ_OUTLINE_ZERO) || (width > k_maxWidthx)) continue;



                /**
                 *----Skip vSeg2 if the horizontal distance H is greater than the
                 *----minimum first pair width found so far (i.e., search for the
                 *----segment pair with the minimum first pair width)
                 */
                if ((minPairWidth > BAZ_OUTLINE_ZERO) && (width > minPairWidth))
                    continue;


                /**
                 *----Skip vSeg2 if it does not sufficiently overlap vSeg
                 */
                {
                    /**
                     *----Determine the vertical overlap between vSeg and vSeg2
                     */
                    ADF_R32 overlap = ADF_MIN(vSeg->maxMerged, vSeg2->maxMerged) -
                    ADF_MAX(vSeg->minMerged, vSeg2->minMerged);


                    /**
                     *----Determine the length of vSeg2
                     */
                    ADF_R32 vSegLength2 = vSeg2->maxMerged - vSeg2->minMerged;


                    /**
                     *----Determine the maximum of vSegLength and vSegLength2
                     */
                    ADF_R32 maxLength = (vSegLength2 > vSegLength) ? vSegLength2 :
                    vSegLength;


                    /**
                     *----Skip vSeg2 if it does not sufficiently overlap vSeg (i.e.,
                     *----the two segments in a segment pair must overlap by at least
                     *----k_overlapRatio)
                     */
                    if (overlap < BAZ_OUTLINE_MUL(k_overlapRatio, maxLength))
                        continue;
                }

                


                /**
                 *----vSeg2 satisfies all of the above criteria and is the new
                 *----optimal paired segment for vSeg. Update the minimum first pair
                 *----width and the optimal paired segment.
                 */
                minPairWidth = width;
                bestPairSeg = vSeg2;
            }

            
            /**
             *----All vertical merged segments in R have been considered as potential
             *----optimal paired segments for vSeg. Determine if an optimal paired
             *----segment for vSeg exists.
             */
            if (bestPairSeg) 
            {


                /**
                 *----An optimal paired segment for vSeg (i.e., bestPairSeg) exists.
                 *----It is possible that bestPairSeg has already been paired with
                 *----another vertical merged segment M during an earlier iteration
                 *----of this for-loop. Determine whether to pair bestPairSeg with
                 *----vSeg or M by choosing the segment pair with the minimum first
                 *----pair width.
                 */
                if ((!(bestPairSeg->pairMin)) || (vSeg->p1->ox >=
                bestPairSeg->pairMin->p1->ox)) 
                {
                
                
                    /**
                     *----Either (1) bestPairSeg is an unpaired segment or (2)
                     *----bestPairSeg is the maximum paired segment in a segment pair
                     *----P1 = {M,bestPairSeg} that is thicker than the segment pair
                     *----P2 = {vSeg,bestPairSeg}. In the former case, form the
                     *----segment pair P2 and update the number of vertical segment
                     *----pairs. In the latter case, delete the existing segment pair
                     *----P1 by setting M's pairMax pointer to NULL and form the
                     *----segment pair P2.
                     */
                    if (bestPairSeg->pairMin) bestPairSeg->pairMin->pairMax = 0;
                    else ++numVSegPairs;
                    vSeg->pairMax = bestPairSeg;
                    bestPairSeg->pairMin = vSeg;
                    
                }
            }
        }
    }


    /**
     *----Store the number of horizontal segment pairs
     */
    *outNumHSegPairs = numHSegPairs;

    
    /**
     *----Store the number of vertical segment pairs
     */
    *outNumVSegPairs = numVSegPairs;
}



static void BAZOutlineSortSegmentPairsDescending (MAZOutlineSegPair *segPairs, ADF_I32
numSegPairs)
{
    /**
     *----Set a pointer to the first segment pair
     */
    MAZOutlineSegPair *sp = segPairs;


    /**
     *----Process each segment pair except for the last segment pair
     */
    ADF_I32 i;
    
    for (i = 0; i < numSegPairs - 1; ++i, ++sp)
    {


        /**
         *----Initialize the minimum original coordinate
         */
        ADF_R32 cMin = sp->cOrig;
        ADF_I32 j;


        /**
         *----Initialize a pointer to the segment pair with the minimum original
         *----coordinate
         */
        MAZOutlineSegPair *minSegPair = sp;


        /**
         *----Set a pointer to the segment pair after sp
         */
        MAZOutlineSegPair *spn = sp + 1;


        /**
         *----Process each segment pair after sp
         */
        for (j = i + 1; j < numSegPairs; ++j, ++spn) {


            /**
             *----If segment pair spn has an original coordinate that is less than
             *----the current minimum original coordinate (i.e., cMin), update cMin
             *----and minSegPair
             */
            if (spn->cOrig > cMin) 
            {
                cMin = spn->cOrig;
                minSegPair = spn;
            }
        }


        /**
         *----If the segment pair with the minimum original coordinate is different
         *----from sp, swap the values in sp and minSegPair
         */
        if (minSegPair != sp)
        {
            MAZOutlineSegPair tmp = *minSegPair;
            *minSegPair = *sp;
            *sp = tmp;
        }
    }
}


/**
 *-----------------------------------------------------------------------------------
 *    Return true (i.e., a non-zero value) if segment pairs s1 and s2 overlap by at
 *    least BAZ_OUTLINE_ONE_FIFTH units in real-valued image coordinates. Return false
 *    (i.e., zero) otherwise.
 *-----------------------------------------------------------------------------------
 */
ADF_INLINE static ADF_I32 BAZOutlineSegPairsOverlap (MAZOutlineSegPair *s1,
MAZOutlineSegPair *s2)
{


    return(!((s1->cMax - BAZ_OUTLINE_TWO < s2->cMin) || (s2->cMax -
    BAZ_OUTLINE_TWO < s1->cMin)));
}



static void BAZOutlineBuildDAGTopDown (MAZOutlineSegPair *segPairs, ADF_I32 numSegPairs)
{
    /**
     *----Set a pointer to the second segment pair (skip the first segment pair
     *----because the first segment pair has the minimum original coordinate and
     *----therefore cannot have any children)
     */
    MAZOutlineSegPair *sp = segPairs + 1;


    /**
     *----Process each segment pair in ascending order
     */
    ADF_I32 i;
    for (i = 1; i < numSegPairs; ++i, ++sp) 
    {


        /**
         *----Set a pointer to the previous segment pair
         */
        MAZOutlineSegPair *spPrev = sp - 1;


        /**
         *----Process the previous segment pairs in descending order
         */
        ADF_I32 j;
        for (j = i - 1; j >= 0; --j, --spPrev)
        {


            /**
             *----Initialize the addChild Boolean to true (i.e., initially assume
             *----that segment pair spPrev will be added as a child to segment pair
             *----sp)
             */
            ADF_I32 addChild = 1;
            ADF_I32 k;


            /**
             *----If the difference between sp's current hinted coordinate and
             *----spPrev's current hinted coordinate is greater than 3 pixels, then
             *----sp and spPrev cannot collide. In this case, exit the inner for-loop
             *----to prevent spPrev from being added as a child of sp, thereby
             *----improving overall runtime efficiency.
             */
            if (spPrev->cHint - sp->cHint > 3) break;


            /**
             *----If segment pairs sp and spPrev do not overlap, proceed to the next
             *----segment pair
             */
            if (!BAZOutlineSegPairsOverlap(sp, spPrev)) continue;


            /**
             *----Segments pairs sp and spPrev overlap. Determine if spPrev overlaps
             *----any of the children of sp. If an overlap occurs, set the addChild
             *----Boolean to false and exit the loop.
             */
            for (k = 0; k < sp->numChildren; ++k) {
                if (BAZOutlineSegPairsOverlap(spPrev, sp->children[k])) {
                    addChild = 0;
                    break;
                }
            }


            /**
             *----If the addChild Boolean is true (i.e., spPrev does not overlap any
             *----of the children of sp), add spPrev as a child of sp
             */
            if (addChild) 
            {

                
                /**
                 *----Add spPrev as a child of sp and update the number of children
                 *----of sp
                 */
                sp->children[sp->numChildren++] = spPrev;
                spPrev->parents[spPrev->numParents++] = sp;


                /**
                 *----If sp has reached the maximum number of children, proceed to
                 *----the next segment pair (i.e., do not add any more children to
                 *----sp)
                 */
                if (sp->numChildren >= MAZ_OUTLINE_MAX_CHILDREN) break;
            }
        }
    }
}





/**
 *-----------------------------------------------------------------------------------
 *    Grid fit (in-place) the y coordinate of each point in each horizontal segment
 *    pair in the specified horizontal segment pair array segPairs. Upon entry,
 *    segPairs is the contiguous array of numSegPairs horizontal segment pairs whose
 *    points will be grid fit in the y direction and pwaf is the pair width adjustment
 *    factor (see the Algorithm Overview section). Upon exit, the OUTLINE_Y_GRID_FIT
 *    bit flag of each point in each horizontal segment pair in the segPairs array will
 *    be set.
 *-----------------------------------------------------------------------------------
 */
static void BAZOutlineGridFitSegmentPairPointsY (MAZOutlineSegPair *segPairs, ADF_I32 numSegPairs)
{
    /**
     *----Set a pointer to the first horizontal segment pair in the segPairs array
     */
    MAZOutlineSegPair *sp = segPairs;


    /**
     *----Perform grid fitting on each horizontal segment pair in the segPairs array
     */
    ADF_I32 i;
    for (i = 0; i < numSegPairs; ++i, ++sp) 
    {


        /**
         *----Set a pointer to the minimum paired segment (i.e., s) in the current
         *----segment pair (i.e., sp)
         */
        MAZOutlineSegment *s = sp->seg;


        /**
         *----Initialize pairWidth to the original pair width of sp
         */
        ADF_R32 pairWidth = sp->width;


        /**
         *----Process each horizontal simple segment in the minimum paired segment.
         *----Determine the hinted y coordinate of each point P in each horizontal
         *----simple segment by adding delta to P's original y coordinate.
         */
        for (; s; s = s->mergeChain) 
        {


            /**
             *----If the first point of s (i.e., p1) has not already been grid fit in
             *----the y direction (i.e., p1's OUTLINE_Y_GRID_FIT bit flag is clear),
             *----then add delta to p1's original y coordinate and set p1's
             *----OUTLINE_Y_GRID_FIT bit flag. Note that the hy element of p1 is
             *----initialized to the original y coordinate of p1 in
             *----BAZDeterminePointsAndContours() (see above).
             */
            if (!(s->p1->flags & OUTLINE_Y_GRID_FIT)) 
            {
                s->p1->hy = sp->cHint;
                s->p1->flags |= OUTLINE_Y_GRID_FIT;
            }


            /**
             *----If the second point of s (i.e., p2) has not already been grid fit
             *----in the y direction (i.e., p2's OUTLINE_Y_GRID_FIT bit flag is
             *----clear), then add delta to p2's original y coordinate and set p2's
             *----OUTLINE_Y_GRID_FIT bit flag. Note that the hy element of p2 is
             *----initialized to the original y coordinate of p2 in
             *----BAZDeterminePointsAndContours() (see above).
             */
            if (!(s->p2->flags & OUTLINE_Y_GRID_FIT)) 
            {
                s->p2->hy = sp->cHint;
                s->p2->flags |= OUTLINE_Y_GRID_FIT;
            }
        }


        /**
         *----Set a pointer to the maximum paired segment of sp
         */
        s = sp->seg->pairMax;



        /**
         *----Process each horizontal simple segment in the maximum paired segment.
         *----Determine the hinted y coordinate of each point P in each horizontal
         *----simple segment by adding delta to P's original y coordinate.
         */
        for (; s; s = s->mergeChain) 
        {


            /**
             *----If the first point of s (i.e., p1) has not already been grid fit in
             *----the y direction (i.e., p1's OUTLINE_Y_GRID_FIT bit flag is clear),
             *----then add delta to p1's original y coordinate and set p1's
             *----OUTLINE_Y_GRID_FIT bit flag. Note that the hy element of p1 is
             *----initialized to the original y coordinate of p1 in
             *----BAZDeterminePointsAndContours() (see above).
             */
            if (!(s->p1->flags & OUTLINE_Y_GRID_FIT)) 
            {
                s->p1->hy = sp->cHint + pairWidth; 
                /* s->p1->hy -= delta; */
                s->p1->flags |= OUTLINE_Y_GRID_FIT;
            }


            /**
             *----If the second point of s (i.e., p2) has not already been grid fit
             *----in the y direction (i.e., p2's OUTLINE_Y_GRID_FIT bit flag is
             *----clear), then add delta to p2's original y coordinate and set p2's
             *----OUTLINE_Y_GRID_FIT bit flag. Note that the hy element of p2 is
             *----initialized to the original y coordinate of p2 in
             *----BAZDeterminePointsAndContours() (see above).
             */
            if (!(s->p2->flags & OUTLINE_Y_GRID_FIT))
            {
                /* s->p2->hy -= delta; */
                s->p2->hy = sp->cHint + pairWidth; 
                s->p2->flags |= OUTLINE_Y_GRID_FIT;
            }
        }
    }
}


/**
 *-----------------------------------------------------------------------------------
 *    Grid fit (in-place) the x coordinate of each point in each vertical segment pair
 *    in the specified vertical segment pair array segPairs. Upon entry, segPairs is
 *    the contiguous array of numSegPairs vertical segment pairs whose points will be
 *    grid fit in the x direction and pwaf is the pair width adjustment factor (see the
 *    Algorithm Overview section). Upon exit, the OUTLINE_X_GRID_FIT bit flag of each
 *    point in each vertical segment pair in the segPairs array will be set.
 *-----------------------------------------------------------------------------------
 */
static void BAZOutlineGridFitSegmentPairPointsX (MAZOutlineSegPair *segPairs, ADF_I32
numSegPairs, ADF_R32 pwaf)
{
    /**
     *----Set a pointer to the first vertical segment pair in the segPairs array
     */
    MAZOutlineSegPair *sp = segPairs;


    /**
     *----Perform grid fitting on each vertical segment pair in the segPairs array
     */
    ADF_I32 i;
    for (i = 0; i < numSegPairs; ++i, ++sp) 
    {


        /**
         *----Set a pointer to the minimum paired segment (i.e., s) in the current
         *----segment pair (i.e., sp)
         */
        MAZOutlineSegment *s = sp->seg;
        

        /**
         *----Initialize pairWidth to the original pair width of sp
         */
        ADF_R32 pairWidth = sp->width;

        


        /**
         *----Process each vertical simple segment in the minimum paired segment.
         *----Determine the hinted x coordinate of each point P in each vertical
         *----simple segment by adding delta to P's original x coordinate.
         */
        for (; s; s = s->mergeChain) 
        {


            /**
             *----If the first point of s (i.e., p1) has not already been grid fit in
             *----the x direction (i.e., p1's OUTLINE_X_GRID_FIT bit flag is clear),
             *----then add delta to p1's original x coordinate and set p1's
             *----OUTLINE_X_GRID_FIT bit flag. Note that the hx element of p1 is
             *----initialized to the original x coordinate of p1 in
             *----BAZDeterminePointsAndContours() (see above).
             */
            if (!(s->p1->flags & OUTLINE_X_GRID_FIT)) 
            {
             
                s->p1->hx = sp->cHint;
                s->p1->flags |= OUTLINE_X_GRID_FIT;
            }


            /**
             *----If the second point of s (i.e., p2) has not already been grid fit
             *----in the x direction (i.e., p2's OUTLINE_X_GRID_FIT bit flag is
             *----clear), then add delta to p2's original x coordinate and set p2's
             *----OUTLINE_X_GRID_FIT bit flag. Note that the hx element of p2 is
             *----initialized to the original x coordinate of p2 in
             *----BAZDeterminePointsAndContours() (see above).
             */
            if (!(s->p2->flags & OUTLINE_X_GRID_FIT))
            {
             
                s->p2->hx =sp->cHint;
                s->p2->flags |= OUTLINE_X_GRID_FIT;
            }
        }


        /**
         *----Set a pointer to the maximum paired segment of sp
         */
        s = sp->seg->pairMax;


        /**
         *----To improve the visual clarity of sp, ensure that the original pair
         *----width is at least 0.5 pixels
         */
        if (pairWidth < BAZ_OUTLINE_ONE_HALF) pairWidth = BAZ_OUTLINE_ONE_HALF;


        /**
         *----If the original pair width is less than 1 pixel, increase the
         *----original pair width by the ppem-dependent pair width adjustment
         *----factor to improve clarity without compromising perceived stroke
         *----weight consistency (see the Algorithm Overview section for details)
         */
        if (pairWidth < BAZ_OUTLINE_ONE)
            pairWidth = BAZ_OUTLINE_MUL(BAZ_OUTLINE_ONE - pwaf, pairWidth) + pwaf;


        
        
        /**
         *----Process each vertical simple segment in the maximum paired segment.
         *----Determine the hinted x coordinate of each point P in each vertical
         *----simple segment by adding delta to P's original x coordinate.
         */
        for (; s; s = s->mergeChain) 
        {


            /**
             *----If the first point of s (i.e., p1) has not already been grid fit in
             *----the x direction (i.e., p1's OUTLINE_X_GRID_FIT bit flag is clear),
             *----then add delta to p1's original x coordinate and set p1's
             *----OUTLINE_X_GRID_FIT bit flag. Note that the hx element of p1 is
             *----initialized to the original x coordinate of p1 in
             *----BAZDeterminePointsAndContours() (see above).
             */
            if (!(s->p1->flags & OUTLINE_X_GRID_FIT)) 
            {
             
                s->p1->hx = sp->cHint + pairWidth; 
                s->p1->flags |= OUTLINE_X_GRID_FIT;
            }


            /**
             *----If the second point of s (i.e., p2) has not already been grid fit
             *----in the x direction (i.e., p2's OUTLINE_X_GRID_FIT bit flag is
             *----clear), then add delta to p2's original x coordinate and set p2's
             *----OUTLINE_X_GRID_FIT bit flag. Note that the hx element of p2 is
             *----initialized to the original x coordinate of p2 in
             *----BAZDeterminePointsAndContours() (see above).
             */
            if (!(s->p2->flags & OUTLINE_X_GRID_FIT)) 
            {
            
                s->p2->hx = sp->cHint + pairWidth; 

                s->p2->flags |= OUTLINE_X_GRID_FIT;
            }
        }                 
    }
}
 

void BAZadjustStroke(ADF_R32 *topval, ADF_R32 *bottomval)
{
    ADF_R32 top = *topval;
    ADF_R32 bottom = *bottomval;

    /* set the middle of the stroke at a center of a pixel */
    ADF_R32 adj;

    adj = (bottom&0xffff0000) - bottom;

    if (adj > BAZ_OUTLINE_ONE_HALF)
        adj -= BAZ_OUTLINE_ONE;
    else if (adj < -BAZ_OUTLINE_ONE_HALF)
        adj += BAZ_OUTLINE_ONE;

    *topval = top + adj;
    *bottomval = bottom + adj;

}

static void BAZOutlineHints(fnt_LocalGraphicStateType *gs, MAZOutlineSegPair *segPairs, ADF_I32 numSegPairs, ADF_I32 yProc)
{

    ADF_I32 i;
    MAZOutlineSegPair *sp = segPairs;


    for (i = 0; i < numSegPairs; ++i, ++sp) 
    {
        ADF_R32 top;
        ADF_R32 bottom;
        ADF_R32 center1;
        ADF_R32 center2;
        

        if (yProc && sp->hintFlags & HF_YLINEALIGNED)
            continue;

        if (i != numSegPairs-1) 
        {
            center1 = (sp->cOrig + sp->cOrig + sp->width)>>1;
            center2 = (sp[1].cOrig + sp[1].cOrig + sp[1].width)>>1;
        

            if (ABS(center1 - center2) < 0x4000 && 
                ABS(sp->width-sp[1].width) < 0x10000 ) 
            {
                if (sp->cMax - sp->cMin > sp[1].cMax - sp[1].cMin) 
                {
                    /* sp is the larger of the two strokes, grid align that one */
                    if (!gs->globalGS->maz_data.isitalic) {
    
                        bottom = sp->cOrig;
                        top = bottom + sp->width;

                        if (yProc || gs->globalGS->maz_data.isrighttoleft)
                            BAZadjustStroke(&bottom,&top);
                        else
                            BAZadjustStroke(&top,&bottom);

                        sp->cHint = bottom;
                        sp[1].cHint = sp[1].cOrig + sp->cHint - sp->cOrig;
                    } 
                    else 
                    {

                        sp->cHint = sp->cOrig;
                        sp[1].cHint = sp[1].cOrig;
                    }
                
                } 
                else 
                {
                    if (!gs->globalGS->maz_data.isitalic) 
                    {
    
                        bottom = sp[1].cOrig;
                        top = bottom + sp[1].width;

                        if (yProc || gs->globalGS->maz_data.isrighttoleft)
                            BAZadjustStroke(&bottom,&top);
                        else
                            BAZadjustStroke(&top,&bottom);

                        sp[1].cHint = bottom;
                        sp->cHint = sp->cOrig + sp[1].cHint - sp[1].cOrig;
                    } 
                    else 
                    {
                        sp->cHint = sp->cOrig;
                        sp[1].cHint = sp[1].cOrig;
                    }
                    
                }
                i++;
                sp++;
                continue;
            }
        } 
        

        if (!gs->globalGS->maz_data.isitalic) 
        {
    
            bottom = sp->cOrig;
            top = bottom + sp->width;

            if (yProc || gs->globalGS->maz_data.isrighttoleft)
                BAZadjustStroke(&bottom,&top);
            else
                BAZadjustStroke(&top,&bottom);

            sp->cHint = bottom;
        }
        else

            sp->cHint = sp->cOrig;

        
    }

    if (numSegPairs != 3)
        return;

    /* handle special case for centering the middle stroke */
    sp = segPairs;

    /* check if originally centered */
    if (ABS(sp[0].cOrig - sp[1].cOrig - sp[1].cOrig + sp[2].cOrig) > 0x4000)
        return;

    if (sp[0].cHint&0xffff)
        return;
    if (sp[1].cHint&0xffff)
        return;
    if (sp[2].cHint&0xffff)
        return;

    /* check if strokes overlap */
    if (!BAZOutlineSegPairsOverlap(&sp[0],&sp[1]))
        return;
    if (!BAZOutlineSegPairsOverlap(&sp[1],&sp[2]))
        return;

    /* check if currently centered */
    if (sp[0].cHint - sp[1].cHint  == sp[1].cHint - sp[2].cHint)
        return;
    {
        ADF_R32 leftspace = sp[1].cHint - (sp[2].cHint + sp[2].width);
        ADF_R32 rightspace = sp[0].cHint - (sp[1].cHint + sp[1].width);

        if (ABS(rightspace-leftspace) < 0x4000 )
            return;

        if (leftspace < rightspace)
            sp[1].cHint += (rightspace-leftspace)>>1;
        else
            sp[1].cHint -= (leftspace-rightspace)>>1;
    }


}



static void BAZOutlineSegmentYInterpolate(fnt_LocalGraphicStateType *gs, MAZOutlineSegPair *segPairs, ADF_I32 numSegPairs)
{
    MAZOutlineSegPair *sp = segPairs;

    ADF_I32 i;
    ADF_I32 j;
    
    ADF_R32 xDiffInv;
    ADF_R32 f;

    for (i = 0; i < numSegPairs; ++i, ++sp) 
    {
        ADF_R32 origmin = gs->globalGS->controlValueTable[0];
        ADF_R32 origmax = gs->globalGS->controlValueTable[3];
        ADF_R32 newmin = gs->globalGS->controlValueTable[1];
        ADF_R32 newmax = gs->globalGS->controlValueTable[4];
        origmin <<= 10;
        origmax <<= 10;
        newmin <<= 10;
        newmax <<= 10;
        if (sp->hintFlags & HF_YLINEALIGNED)
            continue;

        for (j=0; j<gs->globalGS->maz_data.numylines * 3; j+=3) 
        {
            ADF_R32 cvtorig = gs->globalGS->controlValueTable[gs->globalGS->maz_data.cvtstart+j];
            ADF_R32 cvtadj = gs->globalGS->controlValueTable[gs->globalGS->maz_data.cvtstart+j+1];
            
            cvtorig <<= 10;
            cvtadj <<= 10;

            if (sp->cOrig + sp->width < cvtorig &&
                cvtorig-sp->cOrig + sp->width < origmax-sp->cOrig)
            {
                origmax = cvtorig;
                newmax = cvtadj;
            }
            
            if (sp->cOrig > cvtorig &&
                (sp->cOrig - cvtorig < sp->cOrig - origmin)) 
            {
                origmin = cvtorig;
                newmin = cvtadj;
            }
        } 


        xDiffInv = BAZ_OUTLINE_RCP(origmax - origmin);
        f = BAZ_OUTLINE_MUL(newmax - newmin, xDiffInv);
        

        /**
        *----Interpolate each point p between the anchor points
        */
        sp->cOrig = newmin + BAZ_OUTLINE_MUL(f, sp->cOrig - origmin);
        
    }

}
#define TOLERANCE 0x4000
static void BAZOutlineYlineAlign(fnt_LocalGraphicStateType *gs, MAZOutlineSegPair *segPairs, ADF_I32 numSegPairs)
{

    MAZOutlineSegPair *sp = segPairs;


    /**
     *----Perform grid fitting on each horizontal segment pair in the segPairs array
     */
    ADF_I32 i;
    ADF_I32 j;
    for (i = 0; i < numSegPairs; ++i, ++sp) 
    {
        ADF_R32 mindist = TOLERANCE;
      

        for (j=0; j<gs->globalGS->maz_data.numylines*3; j+=3) 
        {
            ADF_R32 cvtorig = gs->globalGS->controlValueTable[gs->globalGS->maz_data.cvtstart+j];
            ADF_R32 cvtadj = gs->globalGS->controlValueTable[gs->globalGS->maz_data.cvtstart+j+1];
            
            cvtorig <<= 10;
            cvtadj <<= 10;
            if (ABS(cvtorig-sp->cOrig) < mindist) 
            {
                mindist = ABS(cvtorig-sp->cOrig);
                sp->cHint = cvtadj;
                sp->hintFlags |= HF_YLINEALIGNED;
            }

            else if (ABS(cvtorig-sp->cOrig-sp->width) < mindist) 
            {
                mindist = ABS(cvtorig-sp->cOrig-sp->width);
                sp->cHint = cvtadj - sp->width;
                sp->hintFlags |= HF_YLINEALIGNED;             
            }
        }
    }
}

static void BAZYlineAlignFreePoints (fnt_LocalGraphicStateType *gs, MAZOutlineContour *contours, ADF_I32 numContours)
{
    
    ADF_I32 i;
    MAZOutlineContour *c = contours;

    
    /**
     *----Process each contour in the contours array
     */
    
    for (i = 0; i < numContours; ++i, ++c) 
    {


    
        /**
         *----Set a pointer to the first point in c. Proceed to the next contour if
         *----the first point is invalid (note that this should never happen).
         */
        MAZOutlinePoint *p = c->points;
        if (!p) continue;


        /**
         *----Process each point in c
         */
        for (;;) 
        {
            ADF_I32 j;


            /**
             *----Determine if the current point (i.e., p) has already been grid fit
             *----in the y direction
             */
            if (!(p->flags & OUTLINE_Y_GRID_FIT)) 
            {
                ADF_R32 mindist = TOLERANCE;

                if (p->next->oy < p->oy && p->oy < p->prev->oy) 
                {
                    p = p->next;
                    if (p == c->points) break;
                    continue;
                }

                if (p->next->oy > p->oy && p->oy > p->prev->oy) 
                {
                    p = p->next;
                    if (p == c->points) break;
                    continue;
                }
                

                for (j=0; j<gs->globalGS->maz_data.numylines * 3; j+=3)
                {
                    ADF_R32 cvtorig = gs->globalGS->controlValueTable[gs->globalGS->maz_data.cvtstart+j];
                    ADF_R32 cvtadj = gs->globalGS->controlValueTable[gs->globalGS->maz_data.cvtstart+j+1];
                    cvtorig <<= 10;
                    cvtadj <<= 10;
                    if (ABS(cvtorig-p->oy) < mindist) 
                    {
                        p->hy  = cvtadj;
                        p->flags |= OUTLINE_Y_GRID_FIT;
                        mindist = ABS(cvtorig-p->oy);
                    }
                } 
            }


            /**
             *----Advance p to the next point in c. Exit the for-loop if p has
             *----returned to the beginning of the circular linked list of points in
             *----c.
             */
            p = p->next;
            if (p == c->points) break;
        } 


    }
}



/**
 *-----------------------------------------------------------------------------------
 *    Grid fit (in-place) the horizontal paired segments and vertical paired segments
 *    contained (implicitly) in the specified contour array contours. Upon entry,
 *    contours is a contiguous array of numContours contours, segPairs is an
 *    uninitialized contiguous working array of at least N segment pairs (where N is
 *    the sum of numHSegPairs and numVSegPairs), numHSegPairs is the number of
 *    horizontal segment pairs contained (implicitly) in contours, numVSegPairs is the
 *    number of vertical segment pairs contained (implicitly) in contours, and ppem is
 *    the number of pixels per em that will be used to perform grid fitting. 
 *-----------------------------------------------------------------------------------
 */
static void BAZGridFitSegmentPairs (fnt_LocalGraphicStateType *gs,MAZOutlineContour *contours, ADF_I32 numContours,
MAZOutlineSegPair *segPairs, ADF_I32 numHSegPairs, ADF_I32 numVSegPairs, ADF_R32 ppemx)
{
    /**
     *----Local variables
     */
    ADF_I32 i;
    MAZOutlineContour *c;
    MAZOutlineSegPair *hsp;
    MAZOutlineSegPair *vsp;
    MAZOutlineSegment *hSeg;
    MAZOutlineSegment *vSeg;
    MAZOutlineSegment *maxPSeg;
    MAZOutlineSegPair *hSegPairs;
    MAZOutlineSegPair *vSegPairs;
    MAZOutlineSegPair *sp;



    /**
     *----Partition the uninitialized segPairs array into horizontal segment pairs
     *----and vertical segment pairs
     */
    hSegPairs = hsp = segPairs;
    vSegPairs = vsp = segPairs + numHSegPairs;


    /**
     *----Process each root contour in the contours array (i.e., skip over internal
     *----contours). Each root contour implicitly represents a radical R. A
     *----MAZOutlineSegPair data structure (i.e., an element of the segPairs array)
     *----is initialized for each horizontal segment pair in R and for each vertical
     *----segment pair in R.
     */
    for (i = 0, c = contours; i < numContours; ++i, ++c) 
    {

        
        /**
         *----Proceed to the next contour if c is an internal contour (i.e., skip
         *----internal contours)
         */
        if (c != c->radical) continue;


        /**
         *----c is a root contour and therefore (implicitly) represents a radical R.
         *----Initialize a MAZOutlineSegPair data structure for each horizontal
         *----minimum paired segment in R. Note that a horizontal minimum paired
         *----segment identifies a horizontal segment pair.
         */
        for (hSeg = c->hMergeSeg; hSeg; hSeg = hSeg->nextMerge) 
        {


            /**
             *----Skip hSeg if hSeg is not the minimum paired segment in a horizontal
             *----segment pair
             */
            if (!(hSeg->pairMax)) continue;


            /**
             *----hSeg is the minimum paired segment in a horizontal segment pair P.
             *----Set a pointer to the maximum paired segment in P.
             */
            maxPSeg = hSeg->pairMax;


            /**
             *----Set the cOrig element of P to the original middle y coordinate of
             *----hSeg
             */
            hsp->cOrig = BAZ_OUTLINE_MUL(BAZ_OUTLINE_ONE_HALF,
            hSeg->minPerpMerged + hSeg->maxPerpMerged);

            


            /**
             *----Set the width element of P to the original pair width of P
             */
            hsp->width = BAZ_OUTLINE_MUL(BAZ_OUTLINE_ONE_HALF, maxPSeg->minPerpMerged
                                         + maxPSeg->maxPerpMerged) - (hsp->cOrig);

            


            /**
             *----Set the cMin element of P to the minimum original x coordinate over
             *----all points in hSeg and maxPSeg
             */
            hsp->cMin = ADF_MIN(hSeg->minMerged, maxPSeg->minMerged);

            
            /**
             *----Set the cMax element of P to the maximum original x coordinate over
             *----all points in hSeg and maxPSeg
             */
            hsp->cMax = ADF_MAX(hSeg->maxMerged, maxPSeg->maxMerged);


            /**
             *----Set the cHint element of P to the floor of cOrig
             */
            /*  hsp->cHint = BAZ_OUTLINE_TRUNC_I32(hsp->cOrig); */
              hsp->cHint = BAZ_OUTLINE_ROUND_R32(hsp->cOrig);
              hsp->cHint = hsp->cHint>>16;


            /**
             *----Set the cHintLow element of P to the "floor" of (cOrig - 0.5). If
             *----cOrig is less than cHint + 1/2, then set cHintLow to cHint - 1.
             *----Otherwise, set cHintLow to cHint.
             */
            if (gs->globalGS->maz_data.MAZmethodHorz == MAZTOPDOWN || 
                gs->globalGS->maz_data.MAZmethodHorz == MAZUSEHINTS) 
                hsp->cHintLow = (hsp->cOrig > (BAZ_OUTLINE_I32_TO_R32(hsp->cHint) +
                BAZ_OUTLINE_ONE_HALF)) ? (hsp->cHint + 1) : (hsp->cHint);
            else
                hsp->cHintLow = (hsp->cOrig < (BAZ_OUTLINE_I32_TO_R32(hsp->cHint) +
                BAZ_OUTLINE_ONE_HALF)) ? (hsp->cHint - 1) : (hsp->cHint);




            /**
             *----Set the canRoundDownSolo and canRoundDownTree Booleans of P to
             *----false
             */
            hsp->canRoundDownSolo = 0;
            hsp->canRoundDownTree = 0;


            /**
             *----Set the numChildren element of P to 0
             */
            hsp->numChildren = 0;
            hsp->numParents = 0;


            /**
             *----Set the seg element of P to the minimum paired segment of P (i.e.,
             *----hSeg)
             */
            hsp->seg = hSeg;

            /* recompute the width of the segment */
            BAZOutlineRecomputeWidthY(hsp);

            /**
             *----Advance the horizontal segment pair pointer to the next
             *----(uninitialized) horizontal segment pair in the segPairs array
             */
            ++hsp;
        }


        /**
         *----Initialize a MAZOutlineSegPair data structure for each vertical minimum
         *----paired segment in R. Note that a vertical minimum paired segment
         *----identifies a vertical segment pair.
         */
        for (vSeg = c->vMergeSeg; vSeg; vSeg = vSeg->nextMerge) 
        {


            /**
             *----Skip vSeg if vSeg is not the minimum paired segment in a vertical
             *----segment pair
             */
            if (!(vSeg->pairMax)) continue;


            /**
             *----vSeg is the minimum paired segment in a vertical segment pair P.
             *----Set a pointer to the maximum paired segment in P.
             */
            maxPSeg = vSeg->pairMax;
            

            /**
             *----Set the cOrig element of P to the original middle x coordinate of
             *----vSeg
             */
            vsp->cOrig = BAZ_OUTLINE_MUL(BAZ_OUTLINE_ONE_HALF,
            vSeg->minPerpMerged + vSeg->maxPerpMerged);
            
            
            /**
             *----Set the width element of P to the original pair width of P
             */
            vsp->width = BAZ_OUTLINE_MUL(BAZ_OUTLINE_ONE_HALF, maxPSeg->minPerpMerged
            + maxPSeg->maxPerpMerged) - (vsp->cOrig);

            

            /**
             *----Set the cMin element of P to the minimum original y coordinate over
             *----all points in vSeg and maxPSeg
             */
            vsp->cMin = ADF_MIN(vSeg->minMerged, maxPSeg->minMerged);


            /**
             *----Set the cMax element of P to the maximum original y coordinate over
             *----all points in vSeg and maxPSeg
             */
            vsp->cMax = ADF_MAX(vSeg->maxMerged, maxPSeg->maxMerged);

            


            /**
             *----Set the cHint element of P to the floor of cOrig
             */
            /* vsp->cHint = BAZ_OUTLINE_TRUNC_I32(vsp->cOrig); */
            vsp->cHint = BAZ_OUTLINE_ROUND_R32(vsp->cOrig);
            vsp->cHint = vsp->cHint>>16;
            

            /**
             *----Set the cHintLow element of P to the "floor" of (cOrig - 0.5). If
             *----cOrig is less than cHint + 1/2, then set cHintLow to cHint - 1.
             *----Otherwise, set cHintLow to cHint.
             */
            if (gs->globalGS->maz_data.MAZmethodVert == MAZTOPDOWN || 
                gs->globalGS->maz_data.MAZmethodVert == MAZUSEHINTS)
                vsp->cHintLow = (vsp->cOrig > (BAZ_OUTLINE_I32_TO_R32(vsp->cHint) +
                BAZ_OUTLINE_ONE_HALF)) ? (vsp->cHint + 1) : (vsp->cHint);
            else
                vsp->cHintLow = (vsp->cOrig < (BAZ_OUTLINE_I32_TO_R32(vsp->cHint) +
                BAZ_OUTLINE_ONE_HALF)) ? (vsp->cHint - 1) : (vsp->cHint);

            
            /**
             *----Set the canRoundDownSolo and canRoundDownTree Booleans of P to
             *----false
             */
            vsp->canRoundDownSolo = 0;
            vsp->canRoundDownTree = 0;

            
            /**
             *----Set the numChildren element of P to 0
             */
            vsp->numChildren = 0;
            vsp->numParents = 0;
            


            /**
             *----Set the seg element of P to the minimum paired segment of P (i.e.,
             *----vSeg)
             */
            vsp->seg = vSeg;

            /* recompute the width of the segment */
            BAZOutlineRecomputeWidthX(vsp);
            


            /**
             *----Advance the vertical segment pair pointer to the next
             *----(uninitialized) vertical segment pair in the segPairs array
             */
            ++vsp;
        }         
    }

    
   sp = hSegPairs;
            
    
    for (i = 0; i < numHSegPairs ; ++i, ++sp) 
    {
        MAZOutlineSegment *s = sp->seg->pairMax;
        sp->hintFlags = 0;
        while(s) 
        {
        
            
            if (s->p1->hintFlags & HF_YDELTA) 
            {
                sp->hintFlags |= HF_YDELTA;
                sp->Ydelta = s->p1->Ydelta;
            }

            
            if (s->p2->hintFlags & HF_YDELTA) 
            {
                sp->hintFlags |= HF_YDELTA;
                sp->Ydelta = s->p2->Ydelta;
            }
            s = s->mergeChain;
        }
        s = sp->seg->pairMax->pairMin;
        while(s) {
        
            
            if (s->p1->hintFlags & HF_YDELTA) 
            {
                sp->hintFlags |= HF_YDELTA;
                sp->Ydelta = s->p1->Ydelta;
            }
             

            if (s->p2->hintFlags & HF_YDELTA)
            {
                sp->hintFlags |= HF_YDELTA;
                sp->Ydelta = s->p2->Ydelta;
            }
            s = s->mergeChain;
        }
    }

    sp = vSegPairs;

    for (i = 0; i < numVSegPairs ; ++i, ++sp) 
    {
        MAZOutlineSegment *s = sp->seg->pairMax;
        sp->hintFlags = 0;
        while(s) {
        
            
            if (s->p1->hintFlags & HF_XDELTA)
            {
                sp->hintFlags |= HF_XDELTA;
                sp->Xdelta = s->p1->Xdelta;
            }

            if (s->p2->hintFlags & HF_XDELTA)
            {
                sp->hintFlags |= HF_XDELTA;
                sp->Xdelta = s->p2->Xdelta;
            }
            s = s->mergeChain;
        } 
        s = sp->seg->pairMax->pairMin;
        
        while(s) {
        
            
            if (s->p1->hintFlags & HF_XDELTA)
            {
                sp->hintFlags |= HF_XDELTA;
                sp->Xdelta = s->p1->Xdelta;
            }

            
            if (s->p2->hintFlags & HF_XDELTA) 
            {
                sp->hintFlags |= HF_XDELTA;
                sp->Xdelta = s->p2->Xdelta;
            }
            s = s->mergeChain;
        }
    }

    /**
    *----Sort the horizontal segment pairs in the hSegPairs array into ascending
    *----order (i.e., sort by non-decreasing original coordinates). Similarly, sort
    *----the vertical segment pairs in the vSegPairs array into ascending order
    *----(i.e., sort by non-decreasing original coordinates).
    */
    
    BAZOutlineSortSegmentPairsDescending(hSegPairs, numHSegPairs);
    

    
    BAZOutlineSortSegmentPairsDescending(vSegPairs, numVSegPairs);
   

    /**
    *----Build a directed acyclic graph (DAG) for horizontal segment pairs in the
    *----hSegPairs array in which each horizontal segment pair is linked to its
    *----horizontal child segment pairs. Similarly, build a DAG for vertical segment
    *----pairs in the vSegPairs array in which each vertical segment pair is linked
    *----to its vertical child segment pairs.
    */
    
    BAZOutlineBuildDAGTopDown(hSegPairs, numHSegPairs);
    
    BAZOutlineBuildDAGTopDown(vSegPairs, numVSegPairs);

    BAZOutlineYlineAlign(gs, hSegPairs, numHSegPairs);

    BAZOutlineSegmentYInterpolate(gs, hSegPairs, numHSegPairs);

    BAZOutlineHints(gs,hSegPairs, numHSegPairs,1);

    BAZOutlineHints(gs,vSegPairs, numVSegPairs,0);



    /**
     *----Grid fit the horizontal segment pairs in the hSegPairs array and the
     *----vertical segment pairs in the vSegPairs array
     */
    {
        /**
         *----Determine the ppem-dependent pair width adjustment factor
         */
        ADF_R32 pwafx = BAZ_OUTLINE_COMPUTE_PWAF(ppemx);
        if (pwafx < BAZ_OUTLINE_ZERO) pwafx = BAZ_OUTLINE_ZERO;
        else if (pwafx > BAZ_OUTLINE_ONE) pwafx = BAZ_OUTLINE_ONE;
        

        /**
         *----Grid fit the y coordinate of each point in each horizontal segment pair
         *----in the hSegPairs array
         */
        BAZOutlineGridFitSegmentPairPointsY(hSegPairs, numHSegPairs);


        /**
         *----Grid fit the x coordinate of each point in each vertical segment pair
         *----in the vSegPairs array
         */
        BAZOutlineGridFitSegmentPairPointsX(vSegPairs, numVSegPairs, pwafx);
    }
}




/* Interpolate contours which have no segments between the segments */
static void BAZInterpolatePointX (fnt_LocalGraphicStateType *gs, MAZOutlinePoint *point, 
        MAZOutlineSegPair *segs, ADF_I32 numSegs)
{

    ADF_R32 xDiffInv;
    ADF_R32 f;
    ADF_R32 origmin, origmax;
    ADF_R32 newmin, newmax;
    ADF_R32 zonetop,zonebottom;



    /**
     *----Set a pointer to the last segment in segs
     */
    MAZOutlineSegPair *sb = segs + (numSegs - 1);
    MAZOutlineSegPair *st = segs;

    
    /**
     *----Determine the maximum and minimum anchor segment
     */
    ADF_I32 i;
    ADF_R32 toler = 0x8000;

    origmin = point->ox;
    origmax = point->ox;
    newmin = point->ox;
    newmax = point->ox;

    /* only interpolate between segments which lie in the same zone as the point.
     * a zone is defined as the space between two ylines.
     */
    zonetop = gs->globalGS->controlValueTable[3];
    zonebottom = gs->globalGS->controlValueTable[0];
    zonetop <<= 10;
    zonebottom <<= 10;
    for (i=0; i<gs->globalGS->maz_data.numylines * 3; i+=3) 
    {
        ADF_R32 cvtorig = gs->globalGS->controlValueTable[ gs->globalGS->maz_data.cvtstart+i ];
        cvtorig <<= 10;
        
        if (cvtorig > point->oy &&  cvtorig < zonetop) 
        {
            zonetop = cvtorig;
        }
        if (cvtorig < point->oy &&  cvtorig > zonebottom)
        {
            zonebottom = cvtorig;
        }
                

    } 



    
    for (i = numSegs - 1; i >= 0; --i, --sb) 
    {
      
        if (sb->cMin > zonetop + toler ||
            sb->cMax < zonebottom - toler)
            continue;
      
        
        if (point->ox < sb->cOrig) 
        {
            origmax = sb->cOrig;
            newmax = sb->cHint;
            
            break;
        }
        if (point->ox < sb->cOrig + sb->width) 
        {
            origmax = sb->cOrig + sb->width;
            newmax = sb->cHint + sb->width;
            
            break;
        }
    }
    for (i = 0; i <numSegs; ++i, ++st) 
    {
        if (st->cMin > zonetop + toler ||
            st->cMax < zonebottom - toler)
            continue;
        
        if (point->ox > st->cOrig + st->width) 
        {
            origmin = st->cOrig + st->width;
            newmin = st->cHint + st->width;
            
            break;
        }
        if (point->ox > st->cOrig) 
        {
            origmin = st->cOrig;
            newmin = st->cHint;
   
            break;
        }
    }
        
    

    


    /* interpolate each point on the contour */

        
    xDiffInv = BAZ_OUTLINE_RCP(origmax - origmin);
    f = BAZ_OUTLINE_MUL(newmax - newmin, xDiffInv);
        


    /**
     *----Interpolate point between the anchor points
     */
    point->hx = newmin + BAZ_OUTLINE_MUL(f, point->ox - origmin);
    
}


/* Interpolate contours which have no segments between the segments */
static void BAZInterpolatePointY (MAZOutlinePoint *point, 
        MAZOutlineSegPair *segs, ADF_I32 numSegs)
{

    ADF_R32 yDiffInv;
    ADF_R32 f;
    ADF_R32 origmin, origmax;
    ADF_R32 newmin, newmax;


    /**
     *----Set a pointer to the last segment in segs
     */
    MAZOutlineSegPair *sb = segs + (numSegs - 1);
    MAZOutlineSegPair *st = segs;
    
    /**
     *----Determine the maximum and minimum anchor segment
     */
    ADF_I32 i;

    origmin = point->oy;
    origmax = point->oy;
    newmin = point->oy;
    newmax = point->oy;

    
    for (i = numSegs - 1; i >= 0; --i, --sb)
    {
        
        if (point->oy < sb->cOrig) 
        {
            origmax = sb->cOrig;
            newmax = sb->cHint;
            break;
        }
        if (point->oy < sb->cOrig + sb->width) 
        {
            origmax = sb->cOrig + sb->width;
            newmax = sb->cHint + sb->width;
            break;
        }
    }
    for (i = 0; i <numSegs; ++i, ++st) 
    {
        
        if (point->oy > st->cOrig + st->width) 
        {
            origmin = st->cOrig + st->width;
            newmin = st->cHint + st->width;
            break;
        }
        if (point->oy > st->cOrig) 
        {
            origmin = st->cOrig;
            newmin = st->cHint;
            break;
        }
    }
        
    

    



    /* interpolate each point on the contour */

            
        
    yDiffInv = BAZ_OUTLINE_RCP(origmax - origmin);
    f = BAZ_OUTLINE_MUL(newmax - newmin, yDiffInv);
        


    /**
     *----Interpolate point between the anchor points
     */
    point->hy = newmin + BAZ_OUTLINE_MUL(f, point->oy - origmin);
    

    
}






/**
 *-----------------------------------------------------------------------------------
 *    Grid fit (in-place) the x-free and y-free points contained (implicitly) in the
 *    specified contour array contours. Upon entry, contours is a contiguous array of
 *    numContours contours.
 *-----------------------------------------------------------------------------------
 */
static ADF_Void BAZGridFitFreePoints (fnt_LocalGraphicStateType *gs, MAZOutlineContour *contours, ADF_I32 numContours,
                                      MAZOutlineSegPair *segPairs, ADF_I32 numHSegPairs, 
                                      ADF_I32 numVSegPairs)
{
    MAZOutlineSegPair *hSegPairs;
    MAZOutlineSegPair *vSegPairs;
    ADF_I32 i;
    MAZOutlineContour *c = contours;
    ADF_R32 usrsb = gs->CE0->oox[gs->CE0->ep[gs->CE0->nc-1]+2];
    ADF_R32 rsb = gs->CE0->ox[gs->CE0->ep[gs->CE0->nc-1]+2];
    ADF_R32 rsbrounded;

    rsb <<= 10;
    rsbrounded = BAZ_OUTLINE_ROUND_R32(rsb);

    hSegPairs = segPairs;
    vSegPairs = segPairs + numHSegPairs;
    /**
     *----Process each contour in the contours array
     */
    
    for (i = 0; i < numContours; ++i, ++c)
    {

        

        /**
         *----Set a pointer to the first point in c. Proceed to the next contour if
         *----the first point is invalid (note that this should never happen).
         */
        MAZOutlinePoint *p = c->points;
        if (!p) continue;

        

        /**
         *----Process each point in c
         */
        for (;;)
        {

            if (!((p->next->oy < p->oy && p->oy < p->prev->oy) ||
                (p->next->oy > p->oy && p->oy > p->prev->oy)) ||
                !((p->next->ox < p->ox && p->ox < p->prev->ox) ||
                (p->next->ox > p->ox && p->ox > p->prev->ox)) ||
                (gs->CE0->onCurve[p->pointNum] &&
                gs->CE0->onCurve[p->next->pointNum]) ||
                (gs->CE0->onCurve[p->pointNum] &&
                gs->CE0->onCurve[p->prev->pointNum])) 
            {
                
                /**
                *----Determine if the current point (i.e., p) has already been grid fit
                *----in the y direction
                */
                if ((p->flags & OUTLINE_Y_GRID_FIT)==0) 
                {
                    BAZInterpolatePointY (p, hSegPairs, numHSegPairs);
                    p->flags |= OUTLINE_Y_GRID_FIT;
                }
                


            }

            
            if (!((p->next->ox < p->ox && p->ox < p->prev->ox) ||
                (p->next->ox > p->ox && p->ox > p->prev->ox)) ||
                (gs->CE0->onCurve[p->pointNum] &&
                gs->CE0->onCurve[p->next->pointNum]) ||
                (gs->CE0->onCurve[p->pointNum] &&
                gs->CE0->onCurve[p->prev->pointNum]))
            {
                /**
                *----Determine if the current point (i.e., p) has already been grid fit
                *----in the x direction
                */
                if ((p->flags & OUTLINE_X_GRID_FIT)==0) 
                {

                    ADF_R32 oox = gs->CE0->oox[p->pointNum];
                    if (rsb != 0 && p->ox - rsb >= 0 && oox - usrsb < 30) 
                    {
                        p->hx = rsbrounded;
                        p->flags |= OUTLINE_X_GRID_FIT;
                    }
                    else if (oox == 0)
                    {
                        p->hx = 0;
                        p->flags |= OUTLINE_X_GRID_FIT;
                    }
                    else 
                    {
                        BAZInterpolatePointX (gs,p, vSegPairs, numVSegPairs);
                        p->flags |= OUTLINE_X_GRID_FIT;
                    }

                }

            }


            /**
             *----Advance p to the next point in c. Exit the for-loop if p has
             *----returned to the beginning of the circular linked list of points in
             *----c.
             */
            p = p->next;
            if (p == c->points) break;
        } 

        
    }
}


/**
 *-----------------------------------------------------------------------------------
 *    ADFAutohintBAZOutlineInternal() performs BAZ alignment zone detection and grid
 *    fitting on the specified sequence of pen commands penCmds representing an
 *    outline-based glyph. Although ADFAutohintBAZOutlineInternal() can be invoked on
 *    any outline-based glyph, its algorithms are designed and optimized for CJK
 *    glyphs. Upon entry, libInst is a valid Saffron library instance, penCmds is the
 *    contiguous array of numPenCmds pen commands to be grid fit, and ppem is the
 *    number of pixels per em that will be used to perform the grid fitting. The
 *    coordinates of each pen command in the penCmds array must be specified in
 *    non-negative real-valued image coordinates. Grid fitting is performed in-place
 *    (i.e., the computed results are written to the x, y, cx, and cy coordinates of
 *    each pen command in the penCmds array, thereby overwriting their original
 *    values). 
 *
 *    Upon exit, outPoints points to a contiguous array of MAZOutlinePoints,
 *    outContours points to a contiguous array of MAZOutlineContours, outHSegs points
 *    to a contiguous array of horizontal simple segments, outVSegs points to a
 *    contiguous array of vertical simple segments, outNumPoints contains the number of
 *    points in the outPoints array, outNumContours contains the number of contours in
 *    the outContours array, outOuterClockwise contains a Boolean that is set to true
 *    if ADFAutohintBAZOutlineInternal() determined that outer contours of the glyph are
 *    oriented clockwise and false otherwise, and outMem points to a contiguous block
 *    of memory that must be freed by the caller. Freeing outMem also frees outPoints,
 *    outContours, outHSegs, and outVSegs, thereby invalidating all five pointers. The
 *    caller should not free outPoints, outContours, outHSegs, or outVSegs directly.
 *
 *    Note that radicals, merged segments, and segment pairs are represented implicitly
 *    by the returned data structures.
 *
 *    ADFAutohintBAZOutlineInternal() returns true (i.e., a non-zero value) on success,
 *    false (i.e., zero) on failure. Upon failure, no changes are made to the
 *    coordinates in the penCmds array and no data is written to outPoints,
 *    outContours, outHSegs, outVSegs, outNumPoints, outNumContours, outOuterClockwise,
 *    or outMem.
 *-----------------------------------------------------------------------------------
 */
ADF_I32 ADFAutohintBAZOutlineInternal (fnt_LocalGraphicStateType *gs, MAZOutlinePoint **outPoints, MAZOutlineContour
    **outContours, MAZOutlineSegment **outHSegs, MAZOutlineSegment **outVSegs, ADF_I32
    *outNumPoints, ADF_I32 *outNumContours, ADF_I32 *outOuterClockwise, 
    MAZOutlineSegPair **outSegPairs, ADF_I32 *outNumHsegPairs, ADF_I32 *outNumVsegPairs)
{
    /**
     *----Initialized local variables
     */
    ADF_I32 numPoints = 0;
    ADF_I32 numContours = 0;
    ADF_I32 outerClockwise;
    ADF_I32 i;
    ADF_R32 ppemx = gs->globalGS->pixelsPerEm;
    ADF_R32 ppemy = gs->globalGS->pixelsPerEm;


    /**
     *----Uninitialized local variables
     */
    ADF_I32 numHSegPairs;
    ADF_I32 numVSegPairs;
    MAZOutlinePoint *points;
    MAZOutlineContour *contours;
    MAZOutlineContour **contourList;
    MAZOutlineSegment *hSegs;
    MAZOutlineSegment *vSegs;
    MAZOutlineSegPair *segPairs;


    if ( !gs->globalGS->identityTransformation )
    {
        ppemx = ppemx * gs->globalGS->cvtStretchX;
        ppemy = ppemy * gs->globalGS->cvtStretchY;
    }
    else
    {
        ppemx <<= 16;
        ppemy <<= 16;
    }



    /**
     *----Partition the single block of memory into arrays to store points, contours,
     *----horizontal simple segments, vertical simple segments, pointers to contours,
     *----and segment pairs
     */
    contours     =    gs->globalGS->maz_data.contours;
    points       =    gs->globalGS->maz_data.points;
    hSegs        =    gs->globalGS->maz_data.hSegs;
    vSegs        =    gs->globalGS->maz_data.vSegs;
    contourList  =    gs->globalGS->maz_data.contourList;
    segPairs     =    gs->globalGS->maz_data.segPairs;



    /**
     *----Process the pen commands to determine points and contours. Points are
     *----represented explicitly by the MAZOutlinePoint data structure and are stored
     *----in the points array. Contours are represented explicitly by the
     *----MAZOutlineContour data structure and are stored in the contours array.
     *----BAZDeterminePointsAndContours() also determines the contour orientation.
     */
    outerClockwise = BAZDeterminePointsAndContours(gs,points, contours, &numPoints, &numContours);


    for (i=0; i<numPoints; i++) 
    {
        points[i].pointNum = i;
        points[i].hintFlags = 0;
    }

    for (i=0; i<gs->globalGS->maz_data.numXdeltas; i++) 
    {
        points[gs->globalGS->maz_data.Xdeltapoint[i]].hintFlags |= HF_XDELTA;
        points[gs->globalGS->maz_data.Xdeltapoint[i]].Xdelta = gs->globalGS->maz_data.Xdeltashift[i];
    }
    for (i=0; i<gs->globalGS->maz_data.numYdeltas; i++)
    {
        points[gs->globalGS->maz_data.Ydeltapoint[i]].hintFlags |= HF_YDELTA;
        points[gs->globalGS->maz_data.Ydeltapoint[i]].Ydelta = gs->globalGS->maz_data.Ydeltashift[i];
    }



    /**
     *----Determine radicals, which are represented implicitly by the
     *----MAZOutlineContour data structure and are stored in the contours array
     */
    BAZDetermineRadicals(contours, numContours, contourList);


    /**
     *----Determine horizontal simple segments separately for each radical.
     *----Similarly, determine vertical simple segments separately for each radical.
     *----Simple segments are represented explicitly by the MAZOutlineSegment data
     *----structure and are stored in the hSegs and vSegs arrays.
     */
    BAZDetermineSimpleSegments(points, numPoints, hSegs, vSegs);


    /**
     *----Determine horizontal merged segments separately for each radical.
     *----Similarly, determine vertical merged segments separately for each radical.
     *----Merged segments are represented implicitly by the MAZOutlineSegment data
     *----structure and are stored in the contours array as linked lists of simple
     *----segments.
     */
    BAZDetermineMergedSegments(contours, numContours);


    /**
     *----Determine horizontal segment pairs separately for each radical. Similarly,
     *----determine vertical segment pairs separately for each radical. Segment pairs
     *----are represented implicitly by the MAZOutlineSegment data structure using
     *----the pairMin and pairMax elements.
     */
    BAZDetermineSegmentPairs(contours, numContours, ppemx, ppemy, outerClockwise,
    &numHSegPairs, &numVSegPairs);


    /**
     *----Grid fit the horizontal and vertical segment pairs. During grid fitting of
     *----the horizontal segment pairs, BAZGridFitSegmentPairs() modifies the y
     *----coordinates of the points in the horizontal segment pairs. Similarly,
     *----during grid fitting of the vertical segment pairs, BAZGridFitSegmentPairs()
     *----modifies the x coordinates of the points in the vertical segment pairs.
     */

    BAZGridFitSegmentPairs(gs,contours, numContours,
    segPairs, numHSegPairs, numVSegPairs, ppemx);



    /**
     *----Grid fit the x-free and y-free points which are represented implicitly in
     *----the contours array
     */
    BAZYlineAlignFreePoints (gs, contours, numContours);

    BAZGridFitFreePoints (gs, contours, numContours,
                          segPairs, numHSegPairs, numVSegPairs);



    /**
     *----Copy the hinted coordinates of the points (which are contained implicitly
     *----in the contours array) to the pen commands array, thereby overwriting the
     *----original coordinates of the pen commands
     */

    for (i=0; i<numPoints; i++) 
    {
        gs->CE0->x[i] = (points[i].hx/*-BAZ_OUTLINE_ONE_HALF*/)>>10;
        gs->CE0->y[i] = (points[i].hy/*-BAZ_OUTLINE_ONE_HALF*/)>>10;
        if (points[i].flags & OUTLINE_Y_GRID_FIT)
            gs->CE0->f[i] |= 2;/*YMOVED;*/
        if (points[i].flags & OUTLINE_X_GRID_FIT)
            gs->CE0->f[i] |= 1;/*XMOVED;*/
    }

    /**
     *---- Apply the MAZDELTAS AFTER
     */
    {
        MAZOutlineSegPair *sp = segPairs;



        for (i = 0; i < numHSegPairs ; ++i, ++sp)
        {
            MAZOutlineSegment *s = sp->seg->pairMax;
            if (!(sp->hintFlags&HF_YDELTA))
                continue;

            while(s) 
            {

                if (!(points[s->p1->pointNum].hintFlags & HF_PROCESSEDYDELTA))
                    gs->CE0->y[s->p1->pointNum] += sp->Ydelta;
                if (!(points[s->p2->pointNum].hintFlags & HF_PROCESSEDYDELTA))
                    gs->CE0->y[s->p2->pointNum] += sp->Ydelta;

                points[s->p1->pointNum].hintFlags |= HF_PROCESSEDYDELTA;
                points[s->p2->pointNum].hintFlags |= HF_PROCESSEDYDELTA;


                s = s->mergeChain;
            }
            s = sp->seg->pairMax->pairMin;
            while(s)
            {

                if (!(points[s->p1->pointNum].hintFlags & HF_PROCESSEDYDELTA))
                    gs->CE0->y[s->p1->pointNum] += sp->Ydelta;
                if (!(points[s->p2->pointNum].hintFlags & HF_PROCESSEDYDELTA))
                    gs->CE0->y[s->p2->pointNum] += sp->Ydelta; 

                points[s->p1->pointNum].hintFlags |= HF_PROCESSEDYDELTA;
                points[s->p2->pointNum].hintFlags |= HF_PROCESSEDYDELTA;

                s = s->mergeChain;
            }
        }

        for (i = 0; i < numVSegPairs ; ++i, ++sp) 
        {
            MAZOutlineSegment *s = sp->seg->pairMax;
            if (!(sp->hintFlags&HF_XDELTA))
                continue;

            while(s)
            {

                if (!(points[s->p1->pointNum].hintFlags & HF_PROCESSEDXDELTA))
                    gs->CE0->x[s->p1->pointNum] += sp->Xdelta;
                if (!(points[s->p2->pointNum].hintFlags & HF_PROCESSEDXDELTA))
                    gs->CE0->x[s->p2->pointNum] += sp->Xdelta;

                points[s->p1->pointNum].hintFlags |= HF_PROCESSEDXDELTA;
                points[s->p2->pointNum].hintFlags |= HF_PROCESSEDXDELTA;


                s = s->mergeChain;
            }
            s = sp->seg->pairMax->pairMin;
            while(s) 
            {

                if (!(points[s->p1->pointNum].hintFlags & HF_PROCESSEDXDELTA))
                    gs->CE0->x[s->p1->pointNum] += sp->Xdelta;
                if (!(points[s->p2->pointNum].hintFlags & HF_PROCESSEDXDELTA))
                    gs->CE0->x[s->p2->pointNum] += sp->Xdelta; 

                points[s->p1->pointNum].hintFlags |= HF_PROCESSEDXDELTA;
                points[s->p2->pointNum].hintFlags |= HF_PROCESSEDXDELTA;


                s = s->mergeChain;
            }

        }
    }


    gs->CE2 = gs->CE0;
    gs->opCode = 0x30;
    BAZOutlineSmooth(gs);
    gs->opCode = 0x31;
    BAZOutlineSmooth(gs);



    /**
     *----Store the pointer to the points array
     */
    *outPoints = points;


    /**
     *----Store the pointer to the contours array
     */
    *outContours = contours;


    /**
     *----Store the pointer to the array of horizontal simple segments
     */
    *outHSegs = hSegs;


    /**
     *----Store the pointer to the array of vertical simple segments
     */
    *outVSegs = vSegs;


    /**
     *----Store the number of points in the points array
     */
    *outNumPoints = numPoints;


    /**
     *----Store the number of contours in the contours array
     */
    *outNumContours = numContours;


    /**
     *----Store the orientation of outer contours
     */
    *outOuterClockwise = outerClockwise;

    *outSegPairs = segPairs;
    *outNumVsegPairs = numVSegPairs;
    *outNumHsegPairs = numHSegPairs;




    /**
     *----Return success
     */
    return(1);
}

/**
 *-----------------------------------------------------------------------------------
 *    ADFAutohintBAZOutline() performs BAZ alignment zone detection and grid fitting on
 *    the specified sequence of pen commands penCmds representing an outline-based
 *    glyph. Although ADFAutohintBAZOutline() can be invoked on any outline-based glyph,
 *    its algorithms are designed and optimized for CJK glyphs. Upon entry, libInst is
 *    a valid Saffron library instance, penCmds is the contiguous array of numPenCmds
 *    pen commands to be grid fit, and ppem is the number of pixels per em that will be
 *    used to perform the grid fitting. The coordinates of each pen command in the
 *    penCmds array must be specified in non-negative real-valued image coordinates.
 *    Grid fitting is performed in-place (i.e., the computed results are written to the
 *    x, y, cx, and cy coordinates of each pen command in the penCmds array, thereby
 *    overwriting their original values). ADFAutohintBAZOutline() returns true (i.e., a
 *    non-zero value) on success, false (i.e., zero) on failure. Upon failure, no
 *    changes are made to the coordinates in the penCmds array.
 *-----------------------------------------------------------------------------------
 */
ADF_I32 ADFAutohintBAZOutline (fnt_LocalGraphicStateType *gs)
{
    /**
     *----Local variables
     */
    
    ADF_I32 numPoints;
    ADF_I32 numContours;
    ADF_I32 outerClockwise;
    MAZOutlinePoint *points;
    MAZOutlineContour *contours;
    MAZOutlineSegment *hSegs;
    MAZOutlineSegment *vSegs;
    MAZOutlineSegPair *segPairs;
    ADF_I32 numHSegPairs;
    ADF_I32 numVSegPairs;
    ADF_R32 xMin;
    ADF_R32 xMax;
    ADF_R32 yMin;
    ADF_R32 yMax;
    ADF_I32 i;

    ADF_I32 result;


    if (gs->CE0->nc == 0)
        return 0;

 
    
    
    xMin = gs->CE0->x[0];
    xMax = gs->CE0->x[0];
    yMin = gs->CE0->y[0];
    yMax = gs->CE0->y[0];
    
    for (i = 1; i <= gs->CE0->ep[gs->CE0->nc-1]; ++i) 
    {
        
        if (xMin > gs->CE0->x[i])
            xMin = gs->CE0->x[i];
        if (xMax < gs->CE0->x[i])
            xMax = gs->CE0->x[i];
        if (yMin > gs->CE0->y[i])
            yMin = gs->CE0->y[i];
        if (yMax < gs->CE0->y[i])
            yMax = gs->CE0->y[i];
    }

    

    /**
     *----Perform BAZ alignment zone detection and grid fitting on the specified
     *----glyph
     */
    result = ADFAutohintBAZOutlineInternal(gs,
    &points, &contours, &hSegs, &vSegs, &numPoints, &numContours, &outerClockwise,
    &segPairs,&numHSegPairs, &numVSegPairs);


    for (i = 0; i <= gs->CE0->ep[gs->CE0->nc-1]; ++i)
    {
        gs->CE0->ox[i] = gs->CE0->x[i];
        gs->CE0->oy[i] = gs->CE0->y[i];
        gs->CE0->f[i] = 0;
    }

    
    /**
     *----Return the Boolean indicating success or failure
     */
    return(result);
}


FS_VOID BAZpositionStems(fnt_LocalGraphicStateType *gs)
{
    int i;

    gs->globalGS->controlValueTable[ 0 ] = gs->GetCVTEntry(gs,0);
    gs->globalGS->controlValueTable[ 1 ] = gs->GetCVTEntry(gs,1);
    gs->globalGS->controlValueTable[ 3 ] = gs->GetCVTEntry(gs,3);
    gs->globalGS->controlValueTable[ 4 ] = gs->GetCVTEntry(gs,4);

    for (i=6; i<gs->globalGS->cvtCount; i+=3) 
    {
        FS_LONG yline = gs->GetCVTEntry(gs,i);

        gs->globalGS->controlValueTable[ i ] = yline;
        gs->globalGS->controlValueTable[ i+1 ] = (yline +32)&0xffffffc0;
    }

}


/**
 *-----------------------------------------------------------------------------------
 *    END: BAZ ALGN ZONE DETECTION AND GRID FITTING FOR OUTLINE-BASED GLYPHS ONLY
 *-----------------------------------------------------------------------------------
 */
#endif


/**
 *-----------------------------------------------------------------------------------
 *    END: IMPLICIT ADFS ONLY
 *-----------------------------------------------------------------------------------
 */
#endif

/**
 *-----------------------------------------------------------------------------------
 *    END: iType ADF Rendering
 *-----------------------------------------------------------------------------------
 */
#endif
